package kawigi.timer;
import javax.swing.*;
import java.util.*;
import java.awt.*;
import java.text.*;
import com.topcoder.client.contestApplet.common.LocalPreferences;

/**
 *	I think seven-segment LED-looking displays are cool.
 *	
 *  This one is rotated 90 degrees and counts down how many points you can still get for submitting the
 *	current problem (you can watch your points slip through your fingers!)  The graphical code comes from
 *	some old game programming I did.
 **/
public class ProblemTimer extends JComponent implements Runnable
{
	/**
	 *	This thread repaints and waits until doomsday.
	 **/
	private Thread anim;
	private HashMap problemTimes;
	private Integer currentProblemID;
	private static int delay;
	private static Color unlit, foreground, background;
	
	/**
	 *	Causes the animation delay and colors in the widget to be reset from the preferences.
	 **/
	public static void resetPrefs()
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		if (prefs.getProperty("kawigi.timer.delay") == null)
			prefs.setProperty("kawigi.timer.delay", "1000");
		delay = prefs.getFontSize("kawigi.timer.delay");
		if (prefs.getProperty("kawigi.timer.foreground") == null)
			prefs.setColor("kawigi.timer.foreground", Color.green);
		foreground = prefs.getColor("kawigi.timer.foreground");
		if (prefs.getProperty("kawigi.timer.background") == null)
			prefs.setColor("kawigi.timer.background", Color.black);
		background = prefs.getColor("kawigi.timer.background");
		if (prefs.getProperty("kawigi.timer.unlit") == null)
			prefs.setColor("kawigi.timer.unlit", new Color(0, 63, 0));
		unlit = prefs.getColor("kawigi.timer.unlit");
	}
	
	/**
	 *	Calls resetDelay() when this class is loaded.
	 **/
	static
	{
		resetPrefs();
	}
	
	/**
	 *	Formats my numbers the way I'm going to display them.
	 **/
	private static DecimalFormat fmt = new DecimalFormat("0000.00");
	
	/**
	 *	Where each number's segments are (the first index is the digit, the second is the segment number,
	 *	as indicated in the comments in the constructor.
	 **/
	private Polygon[][] segments;
	
	/**
	 *	These magic numbers tell me which segments to light up for each digits.  They're each 10-bit bitmasks.
	 **/
	private final int[] segmentkarnaughmaps = {1005, 927, 1019, 877, 325, 881, 892};
	
	/**
	 *	Creates the new widget and initializes the polygons (aren't you glad I only generate those once?)
	 **/
	public ProblemTimer()
	{
		problemTimes = new HashMap();
		//this is for testing offline:
		select(1, 1000);
		anim = new Thread(this);
		anim.start();
		setBackground(Color.black);
		setPreferredSize(new Dimension(25, 100));
		/*
			info on segments array:
			3 digits of 7 segments - indexed [digit index][segment index].
			basically, I'll make it go like this:
				top segment - 0
				top-right segment - 1
				bottom-right segment - 2
				bottom segment - 3
				bottom-left segment - 4
				top-left segment - 5
				middle segment - 6
				
				 <0>
				A   A
				5   1
				V   V
				 <6>
				A   A
				4   2
				V   V
				 <3>
			
			Each segment has an associated "karnaugh map" in that array of constants above.
			The way those work is that the nth least significant digit of each binary number
			determines whether the segment should be lit up for n.  For instance, the magic
			number for segment 4 is 325, which is 0101000101 in binary.  Starting at the end
			of the number, this means that segment 4 should light up for 0, 2, 6 and 8.  Segment
			1's number is 1005, which is 1111101101 in binary.  This means that segment 1 is lit
			up for 0, 2, 3, 5, 6, 7, 8 and 9, because the 0th, 2nd, 3rd, 5th, 6th, 7th and 8th
			least signficant digits are ones.
		*/
		//prototype is a seven-segment digit if it were at 0,0 and right-side up.
		//I'm going to copy it over and offset it to various locations, and also turn it on its side.
		//part of the reason I make them first right-side up is that I'm reusing code from another
		//project ;-)
		Polygon[] prototype = new Polygon[7];
		for (int i=0; i<7; i++)
			prototype[i] = new Polygon();
		//top segment
		prototype[0].addPoint(2,1);
		prototype[0].addPoint(3,0);
		prototype[0].addPoint(8,0);
		prototype[0].addPoint(9,1);
		prototype[0].addPoint(8,2);
		prototype[0].addPoint(3,2);
		//top-right segment
		prototype[1].addPoint(10,2);
		prototype[1].addPoint(11,3);
		prototype[1].addPoint(11,8);
		prototype[1].addPoint(10,9);
		prototype[1].addPoint(9,8);
		prototype[1].addPoint(9,3);
		//bottom-right segment
		prototype[2].addPoint(10,11);
		prototype[2].addPoint(11,12);
		prototype[2].addPoint(11,17);
		prototype[2].addPoint(10,18);
		prototype[2].addPoint(9,17);
		prototype[2].addPoint(9,12);
		//bottom segment
		prototype[3].addPoint(2,19);
		prototype[3].addPoint(3,20);
		prototype[3].addPoint(8,20);
		prototype[3].addPoint(9,19);
		prototype[3].addPoint(8,18);
		prototype[3].addPoint(3,18);
		//bottom-left segment
		prototype[4].addPoint(1,11);
		prototype[4].addPoint(0,12);
		prototype[4].addPoint(0,17);
		prototype[4].addPoint(1,18);
		prototype[4].addPoint(2,17);
		prototype[4].addPoint(2,12);
		//top-left segment
		prototype[5].addPoint(1,2);
		prototype[5].addPoint(2,3);
		prototype[5].addPoint(2,8);
		prototype[5].addPoint(1,9);
		prototype[5].addPoint(0,8);
		prototype[5].addPoint(0,3);
		//middle segment
		prototype[6].addPoint(2,10);
		prototype[6].addPoint(3,9);
		prototype[6].addPoint(8,9);
		prototype[6].addPoint(9,10);
		prototype[6].addPoint(8,11);
		prototype[6].addPoint(3,11);
		segments = new Polygon[6][7];
		for (int i=0; i<6; i++)
		{
			for (int j=0; j<7; j++)
				segments[i][j] = new Polygon();		//initialize empty polygons
		}
		for (int k=0; k<6; k++)
			for (int i=0; i<7; i++)
				for (int j=0; j<prototype[i].npoints; j++)
					segments[k][i].addPoint(prototype[i].ypoints[j]+3, 100-(prototype[i].xpoints[j]+k*15));
	}
	
	/**
	 *	The method run by the thread "anim".  Repeatedly repaints and waits.
	 **/
	public synchronized void run()
	{
		while (Thread.currentThread() == anim)
		{
			repaint();
			try
			{
				wait(delay);
			}
			catch (InterruptedException ex)
			{
				break;
			}
		}
	}
	
	/**
	 *	Draws the seven-segment display with the current point value.
	 **/
	public void paint(Graphics g)
	{
		if (g == null)
			return;
		g.setColor(background);
		g.fillRect(0, 0, getSize().width, getSize().height);
		double points = 0;
		if (currentProblemID != null && problemTimes.containsKey(currentProblemID))
			points = ((ProblemTimingInfo)problemTimes.get(currentProblemID)).currentScore();
		String drawing = fmt.format(points);
		boolean hitpoint = false, nonzero = false;
		for (int i=0; i<drawing.length(); i++)
		{
			if (drawing.charAt(i) == '.')
			{
				g.setColor(foreground);
				g.drawRect(21, 41, 1, 1);
				hitpoint = true;
			}
			else
			{
				int digit = drawing.charAt(i)-'0';
				if (digit != 0)
					nonzero = true;
				if (nonzero)
				{
					for (int j=0; j<7; j++)
					{
						if (((segmentkarnaughmaps[j]>>digit)&1) > 0) //if there is a 1 in the karnaugh map for this digit at this location
							g.setColor(foreground);
						else
							g.setColor(unlit);
						g.fillPolygon(segments[i - (hitpoint? 1 : 0)][j]);
					}
				}
			}
		}
	}
	
	/**
	 *	Changes which problem is being looked at.
	 *	
	 *	If problemID is a new problem, it starts the timer for that problem.  Otherwise, it reloads the data it has for
	 *	when the problem was started (note that <tt>points</tt> is ignored in this case).
	 **/
	public void select(int problemID, double points)
	{
		if (!problemTimes.containsKey(currentProblemID = new Integer(problemID)))
		{
			problemTimes.put(currentProblemID, new ProblemTimingInfo(points, System.currentTimeMillis(), problemID));
		}
			
	}
	
	/**
	 *	Kills the timer thread.
	 **/
	public synchronized void stop()
	{
		anim = null;
		notify();
	}
}