package kawigi.template;
import java.util.*;
import java.io.*;
import java.beans.*;
import java.net.*;

/**
 *	Represents the template of a class that can be extended by a template.
 *	
 *	It's probably best to think of this as a template template.  Even though it
 *	doesn't literally define a template for a template, it's the template for
 *	how a template is interpreted (in fact, it interprets the real templates).
 *	
 *	This class is JavaBean compliant, so that it can be serialized by java's
 *	XMLEncoder (and deserialized by java's XMLDecoder).  Of course, the XMLEncoder
 *	has some serious issues with the complicated system of classloaders involved
 *	TopCoder plugins, so I don't actually use it.
 **/
public class Template implements Serializable
{
	/**
	 *	The name of the template, which should be the name of the class extended by this
	 *	template.
	 **/
	protected String templateName;
	
	/**
	 *	Represents the method that should be overridden by this template.
	 **/
	protected String methodName;
	
	/**
	 *	A list of parameters to the method that is overridden by the template.
	 *	
	 *	Each element should already be of the format <type> <name>.  The type
	 *	and name should also be documented somewhere, as they may be used by
	 *	template writers.  Please do not name any of your parameters "code",
	 *	as the template translation declares and uses "code" as the name of
	 *	the return variable, and this will cause the output code to not compile.
	 **/
	protected String[] parameters;
	
	/**
	 *	Constructs a new Template object with the given values.
	 **/
	public Template(String templateName, String methodName, String[] parameters)
	{
		this.templateName = templateName;
		this.methodName = methodName;
		this.parameters = parameters;
	}
	
	/**
	 *	Constructs a new Template object with initially null properties.
	 **/
	public Template()
	{
	}
	
	/**
	 *	Accessor for <tt>templateName</tt>.
	 **/
	public String getName()
	{
		return templateName;
	}
	
	/**
	 *	Mutator for <tt>templatename</tt>.
	 *	
	 *	<tt>name</tt> should be the name of the class extended by this template.
	 **/
	public void setName(String name)
	{
		templateName = name;
	}
	
	/**
	 *	Accessor for <tt>methodName</tt>.
	 **/
	public String getMethodName()
	{
		return methodName;
	}
	
	/**
	 *	Mutator for <tt>methodName</tt>.
	 **/
	public void setMethodName(String name)
	{
		methodName = name;
	}
	
	/**
	 *	Accessor for <tt>parameters</tt>
	 **/
	public String[] getMethodParameters()
	{
		return parameters;
	}
	
	/**
	 *	Mutator for <tt>parameters</tt>.
	 **/
	public void setMethodParameters(String[] parameters)
	{
		this.parameters = parameters;
	}
	
	/**
	 *	Writes the given template to disk at <folder>/<templateName>.ktt.
	 *	
	 *	The ktt extension is for "Kawigiedit Template Template".
	 **/
	public void write(File folder)
	{
		try
		{
			/*XMLEncoder encoder = new XMLEncoder(*/
			ObjectOutputStream encoder = new ObjectOutputStream(new FileOutputStream(new File(folder, templateName + ".ktt")));
			encoder.writeObject(this);
			encoder.close();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}
	
	/**
	 *	Translates the KawigiEdit Template (KET) code into java code implementing the template.
	 *	
	 *	The new class has the name given in <tt>className</tt> and searches for tag libraries in <tt>directory</tt>.
	 **/
	public String[] translateCode(File directory, String templateCode, String className) throws TemplateSyntaxException
	{
		Set methodsUsed = new HashSet();
		List tagLibs = new ArrayList();
		List lines = new ArrayList();
		lines.add("public class " + className + " extends " + templateName);
		lines.add("{");
		String templine = "\tpublic String " + methodName + "(";
		for (int i=0; i<parameters.length; i++)
		{
			if (i > 0)
				templine += ", ";
			templine += parameters[i];
		}
		templine += ")";
		lines.add(templine);
		lines.add("\t{");
		lines.add("\t\tString code = \"\";");
		insert(directory, lines, methodsUsed, tagLibs, templateCode);
		lines.add("\t\treturn code;");
		lines.add("\t}");
		Iterator it = methodsUsed.iterator();
		while (it.hasNext())
		{
			lines.add("\t");
			String[] methodlines = ((MethodInfo)it.next()).getMethodCode();
			for (int i=0; i<methodlines.length; i++)
				lines.add("\t" + methodlines[i]);
		}
		lines.add("}");
		String[] ret = new String[lines.size()];
		lines.toArray(ret);
		return ret;
	}
	
	/**
	 *	Does the main actual translation of template code into java code.
	 *	
	 *	This code used to be in the <tt>translateCode</tt> method, but it was seperated from it to better afford
	 *	nested templates.
	 **/
	public void insert(File directory, List lines, Set methodsUsed, List tagLibs, String templateCode) throws TemplateSyntaxException
	{
		String templine = "";
		for (int i=0; i<templateCode.length(); i++)
		{
			if (templateCode.charAt(i) == '\n')
			{
				lines.add("\t\tcode += \"" + encode(templine) + "\\n\";");
				templine = "";
			}
			else if (i < templateCode.length()-1 && templateCode.charAt(i) == '<' && templateCode.charAt(i+1) == '%')
			{
				lines.add("\t\tcode += \"" + encode(templine) + "\";");
				templine = "";
				int starti = i;
				i+=3;
				for (; i<templateCode.length() && (templateCode.charAt(i) != '>' || templateCode.charAt(i-1) != '%'); i++)
					;
				String tag = templateCode.substring(starti, i+1);
				if (!tag.endsWith("%>") || tag.length() < 4)
					throw new TemplateSyntaxException("Unclosed action tag!");
				if (tag.charAt(2) == '!')
				{
					String tagbody = tag.substring(3, tag.length()-2).trim();
					if (tagbody.startsWith("include"))
					{
						TagLibrary lib = TagLibrary.read(new File(directory, tagbody.substring(tagbody.indexOf(' ')).trim() + ".tlb"));
						if (lib == null)
							throw new TemplateSyntaxException("Library not found: \"" + tag + "\"");
						tagLibs.add(lib);
					}
					else if (tagbody.startsWith("insert"))
					{
						String newCode = "";
						try
						{
							BufferedReader inFile = new BufferedReader(new FileReader(tagbody.substring(tagbody.indexOf(' ')).trim() + ".ket"));
							String line;
							while ((line = inFile.readLine()) != null)
								newCode += line + "\n";
							insert(directory, lines, methodsUsed, tagLibs, newCode);
						}
						catch (IOException ex)
						{
							throw new TemplateSyntaxException("Couldn't read inserted file: \"" + tag + "\", file: \"" + tagbody.substring(tagbody.indexOf(' ')).trim() + ".ket\"");
						}
					}
					else
						throw new TemplateSyntaxException("Unknown directive tag: \"" + tag + "\"");
				}
				else if (tag.charAt(2) == '=')
				{
					lines.add("\t\tcode += " + tag.substring(3, tag.length()-2) + ";");
				}
				else if (tag.charAt(2) == ':')
				{
					String tagbody = tag.substring(3, tag.length()-2).trim();
					String tagName;
					String[] params;
					if (tagbody.indexOf('(') < 0)
					{
						tagName = tagbody;
						params = new String[0];
					}
					else
					{
						tagName = tagbody.substring(0, tagbody.indexOf('('));
						if (tagbody.indexOf(')') > tagbody.indexOf('('))
						{
							String paramlist = tagbody.substring(tagbody.indexOf('(')+1, tagbody.indexOf(')'));
							String currentParam = "";
							ArrayList tempList = new ArrayList();
							for (int j=0; j<paramlist.length(); j++)
							{
								if (paramlist.charAt(j) == ',')
								{
									tempList.add(currentParam);
									currentParam = "";
								}
								else if (paramlist.charAt(j) == '\"')
								{
									currentParam += paramlist.charAt(j);
									j++;
									for (; j<paramlist.length() && paramlist.charAt(j) != '\"'; j++)
									{
										currentParam += paramlist.charAt(j);
										if (paramlist.charAt(j) == '\\')
										{
											j++;
											currentParam += paramlist.charAt(j);
										}
									}
									if (j == paramlist.length())
										throw new TemplateSyntaxException("Unclosed String literal: \"" + tag + "\"");
								}
								else
									currentParam += paramlist.charAt(j);
							}
							tempList.add(currentParam);
							params = new String[tempList.size()];
							tempList.toArray(params);
						}
						else
							throw new TemplateSyntaxException("Method parameter list not complete: \"" + tag + "\"");
					}
					MethodInfo method = locateMethod(tagLibs, tagName);
					if (method == null)
						throw new TemplateSyntaxException("Couldn't find method in loaded taglibs: \"" + tag + "\", tag name: " + tagName);
					methodsUsed.add(method);
					String[] uselines = method.getCallingCode(params);
					for (int j=0; j<uselines.length; j++)
						lines.add("\t\t" + uselines[j]);
				}
				else
				{
					StringTokenizer tok = new StringTokenizer(tag.substring(2, tag.length()-2), "\n\r");
					while (tok.hasMoreTokens())
						lines.add("\t\t" + tok.nextToken());
				}
			}
			else
				templine += templateCode.charAt(i);
		}
	}
	
	/**
	 *	Resolves the name of the tag in the list of tag libraries.
	 *	
	 *	Returns <tt>null</tt> if the method was not found.
	 **/
	protected MethodInfo locateMethod(List tagLibs, String tagName)
	{
		for (int i=0; i<tagLibs.size(); i++)
		{
			TagLibrary lib = (TagLibrary)tagLibs.get(i);
			if (lib.defines(tagName))
				return lib.getMethod(tagName);
		}
		return null;
	}
	
	/**
	 *	Opens <folder>/<templateClass>.ket and parses it into <folder>/<templateClass>.java.
	 **/
	public void translateFile(File folder, String templateClass) throws IOException, TemplateSyntaxException
	{
		String text = "";
		BufferedReader inFile = new BufferedReader(new FileReader(new File(folder, templateClass + ".ket")));
		String line;
		while ((line = inFile.readLine()) != null)
			text += line + "\n";
		inFile.close();
		
		String[] outlines = translateCode(folder, text, templateClass);
		File f;
		PrintWriter outFile = new PrintWriter(new BufferedWriter(new FileWriter(f = new File(folder, templateClass + ".java"))));
		for (int i=0; i<outlines.length; i++)
			outFile.println(outlines[i]);
		outFile.close();
	}
	
	/**
	 *	Encodes a string to escape new lines, tabs, carriage returns, single and double quotes, and forward slashes.
	 **/
	protected String encode(String startString)
	{
		startString = startString.replaceAll("\\\\", "\\\\\\\\");
		startString = startString.replaceAll("\\n", "\\\\\\n");
		startString = startString.replaceAll("\\t", "\\\\\\t");
		startString = startString.replaceAll("\\r", "\\\\\\r");
		startString = startString.replaceAll("\"", "\\\\\\\"");
		startString = startString.replaceAll("\'", "\\\\\\\'");
		return startString;
	}
	
	/**
	 *	Reads a .ktt file from disk, deserializing it from XML.
	 *	
	 *	Returns null if the file is not found or cannot be read.
	 **/
	public static Template readKTT(File folder, String filename) throws IOException, TemplateSyntaxException
	{
		/*XMLDecoder decoder = new XMLDecoder(*/
		try
		{
			ObjectInputStream decoder = new ObjectInputStream(new FileInputStream(new File(folder, filename)));
			Template t = (Template)decoder.readObject();
			decoder.close();
			return t;
		}
		catch (ClassNotFoundException ex)
		{
			ex.printStackTrace();
			return null;
		}
	}
}

