package kawigi.template;
import java.util.*;
import java.beans.*;
import java.io.*;

/**
 *	Class representing a generic tag library, with methods to quickly look up
 *	tags by name.
 **/
public class TagLibrary implements Serializable
{
	/**
	 *	Maps method names to <tt>MethodInfo</tt> objects.
	 **/
	protected Map map;
	
	/**
	 *	Creates an empty tag library.
	 **/
	public TagLibrary()
	{
		map = new TreeMap();
	}
	
	/**
	 *	Adds the given <tt>MethodInfo</tt> object to the mapping of tags.
	 **/
	public void addMethod(MethodInfo method)
	{
		map.put(method.getMethodName(), method);
	}
	
	/**
	 *	Removes the given tag from the library, if it exists.
	 **/
	public void remove(MethodInfo method)
	{
		map.remove(method.getMethodName());
	}
	
	/**
	 *	Creates a <tt>MethodInfo</tt> object with the given values and adds it to
	 *	the mapping of tags.
	 **/
	public void addMethod(String name, String[] methodCode, String[] callCode)
	{
		map.put(name, new MethodInfo(name, methodCode, callCode));
	}
	
	/**
	 *	Determines if this tag library defines a method with the given name.
	 **/
	public boolean defines(String tagname)
	{
		return map.containsKey(tagname);
	}
	
	/**
	 *	Returns the <tt>MethodInfo</tt> with the given name.
	 **/
	public MethodInfo getMethod(String tagname)
	{
		return (MethodInfo)map.get(tagname);
	}
	
	/**
	 *	Returns the methods mapped in this tag library as an array.
	 **/
	public MethodInfo[] getMethods()
	{
		Collection methods = map.values();
		MethodInfo[] ret = new MethodInfo[methods.size()];
		Iterator it = methods.iterator();
		for (int i=0; i<ret.length; i++)
			ret[i] = (MethodInfo)it.next();
		return ret;
	}
	
	/**
	 *	Erases any existing tags in the library and adds of the elements in <tt>methods</tt>.
	 **/
	public void setMethods(MethodInfo[] methods)
	{
		map = new HashMap();
		for (int i=0; i<methods.length; i++)
			addMethod(methods[i]);
	}
	
	/**
	 *	Returns the number of tags in this library.
	 **/
	public int getSize()
	{
		return map.size();
	}
	
	/**
	 *	Saves this tag library at the given location.
	 *	
	 *	<tt>filename</tt> should preferrably have a .tlb (for Tag LiBrary) extention.
	 **/
	public void save(File filename)
	{
		try
		{
			Iterator it = map.values().iterator();
			PrintWriter outFile = new PrintWriter(new BufferedWriter(new FileWriter(filename)));
			while (it.hasNext())
				((MethodInfo)it.next()).write(outFile);
			outFile.close();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}
	
	/**
	 *	Reads a TagLibrary from disk.
	 **/
	public static TagLibrary read(File filename)
	{
		try
		{
			TagLibrary taglib = new TagLibrary();
			BufferedReader inFile = new BufferedReader(new FileReader(filename));
			MethodInfo meth;
			while ((meth = MethodInfo.read(inFile)) != null)
				taglib.addMethod(meth);
			inFile.close();
			return taglib;
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			return null;
		}
	}
}
