package kawigi.template;
import java.util.*;
import java.io.*;

/**
 *	Represents a tag in a tag library.
 *	
 *	The typical format of this object is that it is composed of a name (which should
 *	be absolutely unique), calling code which gets inserted where the tag is placed,
 *	and method code that gets inserted later, outside of the primary method.  In some
 *	cases the method code is inherited, or a <tt>MethodInfo</tt> object may not even
 *	literally represent a method call.  In these cases, the method code should be an
 *	empty array.
 *	
 *	This class is bean compliant :-)
 **/
public class MethodInfo implements Comparable, Serializable
{
	/**
	 *	The unique name of the tag associated with this object.
	 *	
	 *	Usually it will be the name of the method implemented, if this tag represents
	 *	a method.
	 **/
	protected String methodName;
	
	/**
	 *	The code of the method being called.
	 *	
	 *	For library tags that actually represent a method call, this is where the method
	 *	is defined, with one element in the array for each "line" of code.
	 **/
	protected String[] methodCode;
	
	/**
	 *	A template for the code inserted where this tag is invoked.
	 *	
	 *	If this tag represents a fairly simple method call, it will probably be of the
	 *	format:
	 *	<pre>
	 *		methodName($1,$2,$3);
	 *	</pre>
	 *	There should be one element in the array for each "line" of code that should be
	 *	inserted (in the above example, there is only one line).
	 **/
	protected String[] callingCode;
	
	/**
	 *	Constructs a new MethodInfo with the given name, method code, and calling code.
	 **/
	public MethodInfo(String methodName, String[] methodCode, String[] callingCode)
	{
		this.methodName = methodName;
		this.methodCode = methodCode;
		this.callingCode = callingCode;
	}
	
	/**
	 *	Creates a new instance of MethodInfo with null fields.
	 **/
	public MethodInfo()
	{
	}
	
	/**
	 *	Translates the <tt>callingCode</tt> template to use the parameter list given.
	 *	
	 *	All it does is replace $1 with the first parameter, $2 with the second, etc.
	 *	It would actually make a good div-2 easy problem some day :-)
	 **/
	public String[] getCallingCode(String[] params)
	{
		String[] ret = new String[callingCode.length];
		for (int i=0; i<callingCode.length; i++)
		{
			ret[i] = callingCode[i];
			for (int j=params.length; j>0 ; j--)
				ret[i] = ret[i].replaceAll("\\$" + j, params[j-1]);
		}
		return ret;
	}
	
	/**
	 *	Accessor for <tt>callingCode</tt>.
	 **/
	public String[] getCallingCodeTemplate()
	{
		return callingCode;
	}
	
	/**
	 *	Mutator for <tt>callingCode</tt>.
	 **/
	public void setCallingCodeTemplate(String[] code)
	{
		callingCode = code;
	}
	
	/**
	 *	Accessor for <tt>methodCode</tt>.
	 **/
	public String[] getMethodCode()
	{
		return methodCode;
	}
	
	/** 
	 *	Mutator for <tt>methodCode</tt>.
	 **/	
	public void setMethodCode(String[] code)
	{
		methodCode = code;
	}
	
	/**
	 *	Accessor for <tt>methodName</tt>.
	 **/
	public String getMethodName()
	{
		return methodName;
	}
	
	/**
	 *	Mutator for <tt>methodName</tt>.
	 **/
	public void setMethodName(String name)
	{
		methodName = name;
	}
	
	/**
	 *	Returns a hash code appropriate for hashing this object, so that it can be stored
	 *	dependably in a <tt>HashSet</tt> or used as a key in a <tt>HashMap</tt>.
	 **/
	public int hashCode()
	{
		return methodName.hashCode();
	}
	
	/**
	 *	Returns true if <tt>o</tt> is a <tt>MethodInfo</tt> with the same name as this object.
	 **/
	public boolean equals(Object o)
	{
		return o instanceof MethodInfo && ((MethodInfo)o).getMethodName().equals(getMethodName());
	}
	
	/**
	 *	Returns whether this <tt>MethodInfo</tt> is greater than, equal to, or less than
	 *	another object, by comparing method names lexically.
	 **/
	public int compareTo(Object o)
	{
		if (o instanceof MethodInfo)
			return getMethodName().compareTo(((MethodInfo)o).getMethodName());
		else
			return toString().compareTo(o.toString());
	}
	
	/**
	 *	Returns a string representation of this object.
	 **/
	public String toString()
	{
		return getMethodName();
	}
	
	/**
	 *	Writes this MethodInfo to a stream.
	 **/
	public void write(PrintWriter out)
	{
		out.println("@method name=" + methodName);
		out.println("@calling-code:");
		for (int i=0; i<callingCode.length; i++)
			out.println(callingCode[i]);
		out.println("@method-code:");
		for (int i=0; i<methodCode.length; i++)
			out.println(methodCode[i]);
		out.println("@/method");
	}
	
	/**
	 *	Reads this MethodInfo from a stream.
	 **/
	public static MethodInfo read(BufferedReader in)
	{
		try
		{
			String line = in.readLine();
			if (line != null && line.trim().startsWith("@method"))
			{
				line = line.trim().substring("@method".length()).trim();
				if (line.startsWith("name") && line.indexOf('=') >= 0)
				{
					String name = line.substring(line.indexOf('=')+1).trim();
					ArrayList callingCode = new ArrayList(), methodCode = new ArrayList(), current = null;
					while (!(line = in.readLine()).trim().startsWith("@/method"))
					{
						if (line.trim().startsWith("@calling-code:"))
							current = callingCode;
						else if (line.trim().startsWith("@method-code:"))
							current = methodCode;
						else if (current != null)
							current.add(line);
					}
					String[] callingCodeArray = new String[callingCode.size()];
					callingCode.toArray(callingCodeArray);
					String[] methodCodeArray = new String[methodCode.size()];
					methodCode.toArray(methodCodeArray);
					return new MethodInfo(name, methodCodeArray, callingCodeArray);
				}
				else
				{
					System.err.println("Name not properly specified on the @method line");
					return null;
				}
			}
			else
				return null;
		}
		catch (IOException ex)
		{
			System.err.println("I/O Error reading Tag from Tag Library");
			return null;
		}
	}
}
