package kawigi.language;
import com.topcoder.shared.language.*;
import com.topcoder.shared.problem.*;
import com.topcoder.shared.netCommon.*;
import javax.swing.text.*;
import java.util.*;
import java.io.File;
import kawigi.editor.*;
import kawigi.template.*;
import com.topcoder.client.contestApplet.common.LocalPreferences;

public class ExtendedCPPLanguage extends LanguageContainer implements Templatable
{
	/**
	 *	A singleton instance of the extended C++ language.
	 **/
	public final static ExtendedCPPLanguage EXTENDED_CPP_LANGUAGE = new ExtendedCPPLanguage();
	
	static
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		if (prefs.getProperty("kawigi.language." + EXTENDED_CPP_LANGUAGE.getPropertyClass() + ".filename") == null)
			prefs.setProperty("kawigi.language." + EXTENDED_CPP_LANGUAGE.getPropertyClass() + ".filename", "$PROBLEM$.cpp");
		if (prefs.getProperty("kawigi.language." + EXTENDED_CPP_LANGUAGE.getPropertyClass() + ".compiler") == null)
			prefs.setProperty("kawigi.language." + EXTENDED_CPP_LANGUAGE.getPropertyClass() + ".compiler", "g++ $PROBLEM$.cpp");
		if (prefs.getProperty("kawigi.language." + EXTENDED_CPP_LANGUAGE.getPropertyClass() + ".run") == null)
			prefs.setProperty("kawigi.language." + EXTENDED_CPP_LANGUAGE.getPropertyClass() + ".run", (File.separatorChar == '/' ? "$CWD$/a.out" : "\"$CWD$\\a.exe\""));
		if (prefs.getProperty("kawigi.language." + EXTENDED_CPP_LANGUAGE.getPropertyClass() + ".imports") == null)
			prefs.setProperty("kawigi.language." + EXTENDED_CPP_LANGUAGE.getPropertyClass() + ".imports", "vector,list,map,set,deque,stack,bitset,algorithm,functional,numeric,utility,sstream,iostream,iomanip,cstdio,cmath,cstdlib,ctime");
		TemplateManager.register("C++ Template", EXTENDED_CPP_LANGUAGE);
	}
	
	/**
	 *	Creates a new instance of an extended C++ Language from the given TC language implementation.
	 *	
	 *	If <code>lang</code> is not an instance of CPPLanguage, this object may do weird things, but
	 *	then again, C++ coders tend to know when they want something to malfunction.
	 **/
	public ExtendedCPPLanguage(Language lang)
	{
		super(lang);
	}
	
	/**
	 *	Creates a new instance of the extended C++ language from TC's singleton C++ Language instance.
	 **/
	public ExtendedCPPLanguage()
	{
		super(CPPLanguage.CPP_LANGUAGE);
	}
	
	/**
	 *	Returns at a class and method declaration, with methods that have testing code.
	 **/
	public Skeleton getSkeleton(ProblemComponent comp)
	{
		caret = 0;
		return new Skeleton(getTemplateCode(comp, this), caret);
	}
	
	/**
	 *	Used to set the caret location in the code template.
	 **/
	protected int caret;
	
	/**
	 *	The templatable method for the C++ template
	 **/
	public String getTemplateCode(ProblemComponent comp, Language lang)
	{
		String skeleton = "";
		LocalPreferences prefs = LocalPreferences.getInstance();
		String[] imports = prefs.getProperty("kawigi.language." + getPropertyClass() + ".imports").split(",");
		for (int i=0; i<imports.length; i++)
			skeleton += "#include <" + imports[i] + ">\n";
		skeleton += "\nusing namespace std;\n\n";
		skeleton += "class " + comp.getClassName() + " {\n";
		skeleton += "public:\n\t" + comp.getReturnType(lang.getId()) + " " + comp.getMethodName() + "(";
		DataType[] paramTypes = comp.getParamTypes();
		skeleton += paramTypes[0].getDescriptor(lang);
		for (int i=1; i<paramTypes.length; i++)
			skeleton += ", " + paramTypes[i].getDescriptor(lang);
		skeleton += ");\n};\n\n";
		skeleton += comp.getReturnType(getId()) + " " + comp.getClassName() + "::" + comp.getMethodName() + "(";
		String[] paramNames = comp.getParamNames();
		skeleton += paramTypes[0].getDescriptor(lang) + " " + paramNames[0];
		for (int i=1; i<paramNames.length; i++)
			skeleton += ", " + paramTypes[i].getDescriptor(lang) + " " + paramNames[i];
		skeleton += ") {\n";
		skeleton += "\t\n";
		caret = skeleton.length()-1;
		skeleton += "}\n\n";
		skeleton += generateTestingCode(comp, lang);
		skeleton += "\n//Powered by [KawigiEdit]\n";
		return skeleton;
	}
	
	/**
	 *	Returns the testing code for the problem statement's default testing cases.
	 **/
	protected String generateTestingCode(ProblemComponent comp, Language lang)
	{
		String testcode = "";
		boolean stringType = comp.getReturnType(lang.getId()).toLowerCase().indexOf("string") >= 0;
		TestCase[] cases = comp.getTestCases();
		DataType[] paramTypes = comp.getParamTypes();
		for (int i=0; i<cases.length; i++)
		{
			testcode += "double test" + i + "() {\n";
			String[] input = cases[i].getInput();
			for (int j=0; j<input.length; j++)
				testcode += makeParameter(input[j], paramTypes[j], j);
			testcode += "\t" + comp.getClassName() + " * obj = new " + comp.getClassName() + "();\n";
			testcode += "\tclock_t start = clock();\n";
			testcode += "\t" + comp.getReturnType().getDescriptor(lang) + " my_answer = obj->" + comp.getMethodName() + "(";
			testcode += "p0";
			for (int j=1; j<input.length; j++)
			{
				testcode += ", p" + j;
			}
			testcode += ");\n";
			testcode += "\tclock_t end = clock();\n";
			testcode += "\tdelete obj;\n";
			testcode += "\tcout <<\"Time: \" <<(double)(end-start)/CLOCKS_PER_SEC <<\" seconds\" <<endl;\n";
			testcode += makeParameter(cases[i].getOutput(), comp.getReturnType(), input.length);
			if (comp.getReturnType().getDescriptor(lang).startsWith("vector"))
			{
				testcode += "\tcout <<\"Desired answer: \" <<endl;\n";
				testcode += "\tcout <<\"\\t{ \";\n";
				testcode += "\tif (p" + input.length + ".size() > 0) {\n";
				testcode += "\t\tcout " + (stringType ? "<<\"\\\"\"" : "") + "<<p" + input.length + "[0]" + (stringType ? "<<\"\\\"\"" : "") + ";\n";
				testcode += "\t\tfor (int i=1; i<p" + input.length + ".size(); i++)\n";
				testcode += "\t\t\tcout <<\", " + (stringType ? "\\\"\"" : "\"") + " <<p" + input.length + "[i]" + (stringType ? "<<\"\\\"\"" : "") + ";\n";
				testcode += "\t\tcout <<\" }\" <<endl;\n";
				testcode += "\t}\n";
				testcode += "\telse\n";
				testcode += "\t\tcout <<\"}\" <<endl;\n";
				testcode += "\tcout <<endl <<\"Your answer: \" <<endl;\n";
				testcode += "\tcout <<\"\\t{ \";\n";
				testcode += "\tif (my_answer.size() > 0) {\n";
				testcode += "\t\tcout " + (stringType ? "<<\"\\\"\"" : "") + "<<my_answer[0]" + (stringType ? "<<\"\\\"\"" : "") + ";\n";
				testcode += "\t\tfor (int i=1; i<my_answer.size(); i++)\n";
				testcode += "\t\t\tcout <<\", " + (stringType ? "\\\"\"" : "\"") + " <<my_answer[i]" + (stringType ? "<<\"\\\"\"" : "") + ";\n";
				testcode += "\t\tcout <<\" }\" <<endl;\n";
				testcode += "\t}\n";
				testcode += "\telse\n";
				testcode += "\t\tcout <<\"}\" <<endl;\n";
				testcode += "\tif (my_answer != p" + input.length + ") {\n";
			}
			else
			{
				testcode += "\tcout <<\"Desired answer: \" <<endl;\n";
				testcode += "\tcout <<\"\\t" + (stringType ? "\\\"\"" : "\"") + " << p" + input.length + (stringType ? " <<\"\\\"\"" : "") + " <<endl;\n";
				testcode += "\tcout <<\"Your answer: \" <<endl;\n";
				testcode += "\tcout <<\"\\t" + (stringType ? "\\\"\"" : "\"") + " << my_answer" + (stringType ? "<<\"\\\"\"" : "") + " <<endl;\n";
				testcode += "\tif (p" + input.length + " != my_answer) {\n";
			}
			testcode += "\t\tcout <<\"DOESN'T MATCH!!!!\" <<endl <<endl;\n";
			testcode += "\t\treturn -1;\n";
			testcode += "\t}\n";
			testcode += "\telse {\n";
			testcode += "\t\tcout <<\"Match :-)\" <<endl <<endl;\n";
			testcode += "\t\treturn (double)(end-start)/CLOCKS_PER_SEC;	//I want to eventually turn this into a time of some kind.\n";
			testcode += "\t}\n";
			testcode += "}\n";
		}
		return testcode;
	}
	
	/**
	 *	Returns a string in the format <type> <name>, <type> <name>, etc.
	 **/
	protected String makeParamList(DataType[] types, String[] names)
	{
		String ret = "";
		for (int i=0; i<types.length; i++)
		{
			if (i > 0)
				ret += ", ";
			ret += types[i].getDescriptor(this) + " " + names[i];
		}
		return ret;
	}
	
	/**
	 *	Returns a string in the format <type>, <type>, etc.
	 **/
	protected String makeParamTypeList(DataType[] types)
	{
		String ret = "";
		for (int i=0; i<types.length; i++)
		{
			if (i > 0)
				ret += ", ";
			ret += types[i].getDescriptor(this);
		}
		return ret;
	}
	
	/**
	 *	Declares an array of the same type as a vector.
	 **/
	protected String toArray(String type, String name)
	{
		String t = getType(type);
		return t + " " + name + "[]";
	}
	
	/**
	 *	Gets the underlying type contained in a vector declaration.
	 **/
	protected String getType(String vectortype)
	{
		return vectortype.substring(vectortype.indexOf('<') + 1, vectortype.lastIndexOf('>'));
	}
	
	/**
	 *	Declares a variable called p&lt;number&gt; from the type and value given.
	 **/
	protected String makeParameter(String param, DataType type, int number)
	{
		if (type.getDescriptor(this).startsWith("vector"))
		{
			String decl = "\t" + toArray(type.getDescriptor(this), "t" + number) + " = ";
			String t = getType(type.getDescriptor(this));
			if (t.equals("double") || t.equals("float"))
			{
				StringTokenizer tok = new StringTokenizer(param, "{}, ");
				decl += "{ ";
				while (tok.hasMoreTokens())
				{
					String token = tok.nextToken();
					if (token.indexOf('.') < 0)
						token += ".0";
					if (t.equals("float"))
						token += "F";
					if (tok.hasMoreTokens())
						token += ", ";
					decl += token;
				}
				decl += " }";
			}
			else if (t.equals("long long"))
			{
				StringTokenizer tok = new StringTokenizer(param, "{}, ");
				decl += "{ ";
				while (tok.hasMoreTokens())
				{
					String token = tok.nextToken();
					decl += token + "LL";
					if (tok.hasMoreTokens())
						decl += ", ";
				}
				decl += " }";
			}
			else
				decl += param;
			decl += ";\n";
			decl += "\t" + type.getDescriptor(this) + " p" + number + "(t" + number + ", t" + number + "+sizeof(t" + number + ")/sizeof(" + getType(type.getDescriptor(this)) + "));\n";
			return decl;
		}
		else
		{
			String ret = "\t" + type.getDescriptor(this) + " p" + number + " = " + param;
			if (type.getDescriptor(this).equals("double") || type.getDescriptor(this).equals("float"))
			{
				if (ret.indexOf('.') < 0)
					ret += ".0";
				if (type.getDescriptor(this).equals("float"))
					ret += "F";
			}
			else if (type.getDescriptor(this).equals("long long"))
				ret += "LL";
			ret += ";\n";
			return ret;
		}
	}
	
	/**
	 *	Returns the class of the C++ implementation of a View
	 **/
	public Class getViewClass()
	{
		return CPPView.class;
	}
	
	/**
	 *	Overrides the property class in LanguageContainer so that it only has alphanumeric characters.
	 **/
	public String getPropertyClass()
	{
		return "cpp";
	}
	
	/**
	 *	Returns the command that should be used to compile the file.
	 *	
	 *  Ideally, this won't be platform-specific, but realistically, some people use different
	 *	compilers than others completely.  It may be possible at some point to allow this to be
	 *	configured.
	 **/
	public String getCompileCommand(ProblemComponent comp)
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		String propname = "kawigi.language." + getPropertyClass() + ".compiler";
		String command;
		if (prefs.getProperty(propname) != null)
			command = prefs.getProperty(propname);
		else
			prefs.setProperty(propname, command = "g++ $PROBLEM$.cpp");
		command = command.replaceAll("\\$PROBLEM\\$", comp.getClassName());
		return command;
	}
	
	/**
	 *	Returns the command that should be used to run the program and test cases.
	 *	
	 *	Again, some care will need to be taken to make this part platform-happy.
	 **/
	public String getRunCommand(ProblemComponent comp, File cwd)
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		String propname = "kawigi.language." + getPropertyClass() + ".run";
		String def;
		if (File.separatorChar == '/')
			def = "$CWD$/a.out";
		else
			def = "\"$CWD$\\a.exe\"";
		String command;
		if (prefs.getProperty(propname) != null)
			command = prefs.getProperty(propname);
		else
			prefs.setProperty(propname, command = def);
		command = command.replaceAll("\\$CWD\\$", cwd.getPath().replaceAll("\\\\", "\\\\\\\\"));
		command = command.replaceAll("\\$PROBLEM\\$", comp.getClassName());
		return command;
	}
	
	/**
	 *	Returns the filename that the source file should be saved as.
	 *	
	 *	You may assume that this is the actual filename used when creating the compile and run
	 *	commands.
	 **/
	public String getFileName(ProblemComponent comp)
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		String propname = "kawigi.language." + getPropertyClass() + ".filename";
		String command;
		if (prefs.getProperty(propname) != null)
			command = prefs.getProperty(propname);
		else
			prefs.setProperty(propname, command = "$PROBLEM$.cpp");
		command = command.replaceAll("\\$PROBLEM\\$", comp.getClassName());
		return command;
	}
	
	/**
	 *	Returns the KTT template representing how this class is templated.
	 **/
	public Template getTemplate()
	{
		return new Template(getClass().getName(), "getTemplateCode", new String[]{"com.topcoder.shared.problem.ProblemComponent comp", "com.topcoder.shared.language.Language lang"});
	}
	
	/**
	 *	Returns a library of tags that may come in useful for writing templates extending this class.
	 **/
	public TagLibrary getNativeTagLibrary()
	{
		TagLibrary taglib = new TagLibrary();
		taglib.addMethod("class-name", new String[0], new String[]{"code += comp.getClassName();"});
		taglib.addMethod("method-name", new String[0], new String[]{"code += comp.getMethodName();"});
		taglib.addMethod("return-type", new String[0], new String[]{"code += comp.getReturnType().getDescriptor(lang);"});
		taglib.addMethod("param-type-list", new String[0], new String[]{"code += makeParamTypeList(comp.getParamTypes());"});
		taglib.addMethod("param-list", new String[0], new String[]{"code += makeParamList(comp.getParamTypes(), comp.getParamNames());"});
		taglib.addMethod("testing-code", new String[0], new String[]{"code += generateTestingCode(comp, lang);"});
		taglib.addMethod("set-caret", new String[0], new String[]{"caret = code.length();"});
		return taglib;
	}
	
	/**
	 *	Returns the text of a default template.
	 **/
	public String getDefaultTemplateCode()
	{
		return	"<%!include " + getClass().getName() + " %>\n"
			+	"#include <vector>\n"
			+	"#include <list>\n"
			+	"#include <map>\n"
			+	"#include <set>\n"
			+	"#include <deque>\n"
			+	"#include <stack>\n"
			+	"#include <bitset>\n"
			+	"#include <algorithm>\n"
			+	"#include <functional>\n"
			+	"#include <numeric>\n"
			+	"#include <utility>\n"
			+	"#include <sstream>\n"
			+	"#include <iostream>\n"
			+	"#include <iomanip>\n"
			+	"#include <cstdio>\n"
			+	"#include <cmath>\n"
			+	"#include <cstdlib>\n"
			+	"#include <ctime>\n\n"
			+	"using namespace std;\n\n"
			+	"class <%:class-name%> {\n"
			+	"public:\n"
			+	"\t<%:return-type%> <%:method-name%>(<%:param-type-list%>);\n"
			+	"}\n\n"
			+	"<%:return-type%> <%:method-name%>(<%:param-list%>) {\n"
			+	"\t<%:set-caret%>\n"
			+	"}\n\n"
			+	"<%:testing-code%>\n"
			+	"//Powered by [KawigiEdit]\n";
	}
}
