package kawigi.config;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.undo.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;
import java.net.*;
import kawigi.editor.*;
import kawigi.template.*;
import kawigi.util.*;
import kawigi.language.*;
import com.topcoder.client.contestApplet.common.LocalPreferences;

/**
 *	Editor panel for templates used from the KawigiEdit config panel.
 *	
 *	I consider this a somewhat more advanced feature, but hopefully not one that people can't figure out.
 *	It would be a much more 'advanced' feature (where 'advanced' means 'impossible to figure out') if this
 *	editor weren't in place, so this is my effort at useability in the template system.
 **/
public class TemplateEditor extends JPanel implements KeyListener, UndoableEditListener, ActionListener
{
	private NoWrapJTextPane editorPane;
	private JButton compileButton, saveButton, saveAsButton, openButton, installTemplatesButton, changeTemplateButton;
	private UndoManager undo;
	private File location, templateLocation;
	
	/**
	 *	Creates the Template editor panel with all the components that the user can interact with.
	 **/
	public TemplateEditor()
	{
		setLayout(new BorderLayout());
		editorPane = new NoWrapJTextPane();
		editorPane.setContentType("text/kawigiedit-template");
		LocalPreferences prefs = LocalPreferences.getInstance();
		editorPane.setBackground(prefs.getColor("kawigi.editor.background"));
		editorPane.setForeground(prefs.getColor("kawigi.editor.foreground"));
		editorPane.setCaretColor(prefs.getColor("kawigi.editor.foreground"));
		editorPane.setFont(new Font(prefs.getProperty("kawigi.editor.font"), Font.PLAIN, prefs.getFontSize("kawigi.editor.font.size")));
		editorPane.addKeyListener(this);
		editorPane.getStyledDocument().addUndoableEditListener(this);
		editorPane.setDragEnabled(true);
		undo = new UndoManager();
		LineNumbers lines = new LineNumbers(editorPane);
		JViewport rowViewport = new JViewport();
		rowViewport.setView(lines);
		rowViewport.setBackground(editorPane.getBackground());
		JScrollPane scroll = new JScrollPane(editorPane, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		scroll.setRowHeader(rowViewport);
		JPanel corner = new JPanel();
		corner.setBackground(editorPane.getBackground());
		scroll.setCorner(JScrollPane.LOWER_LEFT_CORNER, corner);
		add(scroll);
		JPanel buttonPanel = new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
		
		openButton = new JButton("Open");
		openButton.addActionListener(this);
		openButton.setMnemonic(KeyEvent.VK_O);
		openButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(openButton);
		
		saveButton = new JButton("Save");
		saveButton.addActionListener(this);
		saveButton.setMnemonic(KeyEvent.VK_S);
		saveButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(saveButton);
		
		saveAsButton = new JButton("Save As");
		saveAsButton.addActionListener(this);
		saveAsButton.setMnemonic(KeyEvent.VK_A);
		saveAsButton.setDisplayedMnemonicIndex(5);
		buttonPanel.add(saveAsButton);
		
		compileButton = new JButton("Compile");
		compileButton.addActionListener(this);
		compileButton.setMnemonic(KeyEvent.VK_C);
		compileButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(compileButton);
		
		changeTemplateButton = new JButton("Change Template");
		changeTemplateButton.addActionListener(this);
		changeTemplateButton.setMnemonic(KeyEvent.VK_T);
		changeTemplateButton.setDisplayedMnemonicIndex(7);
		buttonPanel.add(changeTemplateButton);
		
		installTemplatesButton = new JButton("Install Templates");
		installTemplatesButton.addActionListener(this);
		installTemplatesButton.setMnemonic(KeyEvent.VK_I);
		installTemplatesButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(installTemplatesButton);
		
		add(buttonPanel, BorderLayout.SOUTH);
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	Specifically listens for ctrl+z and ctrl+y for undo and redo, respectively.
	 **/
	public void keyPressed(KeyEvent e)
	{
		if (e.getKeyCode() == KeyEvent.VK_Z && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
		{
			try
			{
				undo.undo();
			}
			catch (CannotUndoException ex)
			{
			}

		}
		if (e.getKeyCode() == KeyEvent.VK_Y && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
		{
			try
			{
				undo.redo();
			}
			catch (CannotRedoException ex)
			{
			}
		}
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 **/
	public void keyReleased(KeyEvent e)
	{
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	This handles the auto-indent.
	 **/
	public void keyTyped(KeyEvent e)
	{
		if (e.getKeyChar() == '\n')
		{
			String text = editorPane.getText().replaceAll("\\r", "");
			String line = "";
			String indentation = "";
			int lines = 0;
			for (int ind = -1; ind < editorPane.getCaretPosition(); ind = text.indexOf('\n', ind+1))
			{
				if (ind < 0 && lines != 0)
					break;
				lines++;
				indentation = (line.trim().length() == 0) ? line : line.substring(0, line.indexOf(line.trim()));
				line = (text.indexOf('\n', ind+1) < 0) ? text.substring(ind+1) : text.substring(ind+1, text.indexOf('\n', ind+1));
			}
			if ((e.getModifiersEx()&InputEvent.SHIFT_DOWN_MASK) != 0)
				indentation = "\n" + indentation;
			editorPane.replaceSelection(indentation);
		}
		editorPane.repaint();
	}
	
	/**
	 *	Stores away edits so they can be undone later.
	 **/
	public void undoableEditHappened(UndoableEditEvent e)
	{
		undo.addEdit(e.getEdit());
	}
	
	/**
	 *	Processes actions from the buttons.
	 **/
	public void actionPerformed(ActionEvent e)
	{
		if (e.getSource() == compileButton)
		{
			save(false);
			if (templateLocation == null)
			{
				JOptionPane.showMessageDialog(this, "You must first select a template template for this template.", "Yes, I know my terminology sucks.", JOptionPane.INFORMATION_MESSAGE);
				promptForTemplate();
			}
			if (templateLocation == null)
				return;
			String templateName = location.getName().substring(0, location.getName().indexOf('.'));
			File folder = location.getParentFile();
			try
			{
				Template templ = Template.readKTT(folder, templateLocation.getName());
				templ.translateFile(folder, templateName);
				System.out.println("Finished the translation");
				LocalPreferences prefs = LocalPreferences.getInstance();
				String classpath = "." + File.pathSeparator + "ContestApplet.jar";
				boolean found = false;
				for (int i=1; i<= Integer.parseInt(prefs.getProperty("editor.numplugins")); i++)
				{
					if (prefs.getProperty("editor." + i + ".entrypoint").equals("kawigi.KawigiEdit"))
					{
						found = true;
						if (prefs.getProperty("editor." + i + ".classpath") != null && prefs.getProperty("editor." + i + ".classpath").length() > 0)
							classpath += File.pathSeparator + prefs.getProperty("editor." + i + ".classpath");
					}
				}
				if (!found)
				{
					JOptionPane.showMessageDialog(this, "The settings for the KawigiEdit plugin were not found.  You may have to exit the config dialog and hit \"Save\" and try again.", "Error", JOptionPane.ERROR_MESSAGE);
					return;
				}
				if (prefs.getProperty("com.topcoder.client.contestApplet.common.LocalPreferences.commonclasspath") != null && prefs.getProperty("com.topcoder.client.contestApplet.common.LocalPreferences.commonclasspath").length() > 0)
					classpath += File.pathSeparator + prefs.getProperty("com.topcoder.client.contestApplet.common.LocalPreferences.commonclasspath");
				String command = prefs.getProperty("kawigi.language.java.compiler");
				if (command.startsWith("\""))
					command = command.substring(0, command.indexOf(' ', command.indexOf('\"', 1)));
				else
					command = command.substring(0, command.indexOf(' '));
				//if (File.separatorChar == '/')
				//	classpath = classpath.replaceAll(" ", "\\\\ ");
				//else
				//	classpath = "\"" + classpath + "\"";
				//command += " -classpath " + classpath + " " + new File(folder, templateName + ".java").getAbsolutePath();
				try
				{
					Process p = Runtime.getRuntime().exec(new String[]{command, "-classpath", classpath, new File(folder, templateName + ".java").getAbsolutePath()}, null, folder);
					JTextArea errorPane = new JTextArea();
					ProcessContainer proc = new ProcessContainer(p, errorPane, false);
					proc.start();
					proc.waitFor();
					if (proc.endVal() != 0)
					{
						System.err.println(errorPane.getText());
						JOptionPane.showMessageDialog(this, "Compile process exited with non-zero status.  View the Java Console for details.", "Error", JOptionPane.ERROR_MESSAGE);
					}
					else if (JOptionPane.showConfirmDialog(this, "Template Compiled!  Set template usage now?", "Success!", JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION)
					{
						try
						{
							Templatable tempable = (Templatable)Class.forName(templ.getName()).newInstance();
							if (tempable instanceof ExtendedLanguage)
								prefs.setProperty("kawigi.language." + ((ExtendedLanguage)tempable).getPropertyClass() + ".override", templateName);
							else
								System.err.println("Templatable not language");
						}
						catch (ClassNotFoundException ex)
						{
							ex.printStackTrace();
						}
					}
				}
				catch (Exception ex)
				{
					JOptionPane.showMessageDialog(this, "Error starting the compile process: " + ex.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
				}
			}
			catch (TemplateSyntaxException ex)
			{
				JOptionPane.showMessageDialog(this, "Template Syntax Error: " + ex.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
			}
			catch (IOException ex)
			{
				JOptionPane.showMessageDialog(this, "Couldn't open template template: " + ex.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
			}
		}
		else if (e.getSource() == saveButton)
		{
			save(false);
		}
		else if (e.getSource() == saveAsButton)
		{
			save(true);
		}
		else if (e.getSource() == openButton)
		{
			LocalPreferences prefs = LocalPreferences.getInstance();
			JFileChooser chooser = new JFileChooser(new File(prefs.getProperty("kawigi.localpath")));
			chooser.setFileFilter(new GenericFileFilter("KawigiEdit Templates (*.ket)", "ket"));
			if (chooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION)
			{
				location = chooser.getSelectedFile();
				if (location.exists())
				{
					
					BufferedReader inFile = null;
					try
					{
						inFile = new BufferedReader(new FileReader(location));
						String stuff = "";
						String line;
						while ((line = inFile.readLine()) != null)
							stuff += line + "\n";
						editorPane.setText(stuff);
						inFile.close();
					}
					catch (IOException ex)
					{
						if (inFile != null)
							try
							{
								inFile.close();
							}
							catch (IOException ex2)
							{
							}
					}
				}
			}
		}
		else if (e.getSource() == installTemplatesButton)
		{
			Iterator it = TemplateManager.getKeys().iterator();
			LocalPreferences prefs = LocalPreferences.getInstance();
			File folder = new File(prefs.getProperty("kawigi.localpath"));
			while (it.hasNext())
			{
				Templatable tempable = TemplateManager.getTemplatable((String)it.next());
				try
				{
					Template templ = tempable.getTemplate();
					templ.write(folder);
					TagLibrary tlib = tempable.getNativeTagLibrary();
					if (tlib != null)
						tlib.save(new File(folder, templ.getName() + ".tlb"));
					String defaultCode = tempable.getDefaultTemplateCode();
					String ketName = templ.getName();
					if (ketName.indexOf('.') >= 0)
						ketName = ketName.substring(ketName.lastIndexOf('.')+1);
					ketName ="Default" + ketName + ".ket";
					PrintWriter outFile = new PrintWriter(new BufferedWriter(new FileWriter(new File(folder, ketName))));
					outFile.println(defaultCode);
					outFile.close();
				}
				catch (IOException ex)
				{
					JOptionPane.showMessageDialog(this, "Error installing templates for " + tempable.getClass().getName(), "Error", JOptionPane.ERROR_MESSAGE);
				}
			}
			try
			{
				BufferedOutputStream outFile = new BufferedOutputStream(new FileOutputStream(new File(folder, "ContestApplet.jar")));
				BufferedInputStream inFile = new BufferedInputStream(new URL("http://www.topcoder.com/contest/classes/ContestApplet.jar").openStream());
				int b;
				while ((b = inFile.read()) != -1)
					outFile.write(b);
				outFile.close();
				inFile.close();
			}
			catch (IOException ex)
			{
				JOptionPane.showMessageDialog(this, "Error downloading the contest applet.", "Error", JOptionPane.ERROR_MESSAGE);
			}
			JOptionPane.showMessageDialog(this, "Finished installing custom template system", "Finished", JOptionPane.INFORMATION_MESSAGE);
		}
		else if (e.getSource() == changeTemplateButton)
		{
			promptForTemplate();
		}
	}
	
	/**
	 *	Saves the current file.
	 *	
	 *	Prompts the user for the file to save to if necessary, or if prompt is true.
	 **/
	public void save(boolean prompt)
	{
		if (prompt || location == null)
		{
			LocalPreferences prefs = LocalPreferences.getInstance();
			JFileChooser chooser = new JFileChooser(new File(prefs.getProperty("kawigi.localpath")));
			chooser.setFileFilter(new GenericFileFilter("KawigiEdit Templates (*.ket)", "ket"));
			if (chooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION)
			{
				location = chooser.getSelectedFile();
				templateLocation = null;
			}
			else
				return;
		}
		PrintWriter outFile = null;
		try
		{
			outFile = new PrintWriter(new BufferedWriter(new FileWriter(location)));
			outFile.println(editorPane.getText());
			outFile.close();
		}
		catch (IOException ex)
		{
			if (outFile != null)
				outFile.close();
		}
	}
	
	/**
	 *	Prompts for and normally sets the location of the current template for which this template is being written.
	 **/
	public void promptForTemplate()
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		JFileChooser chooser = new JFileChooser(new File(prefs.getProperty("kawigi.localpath")));
		chooser.setDialogTitle("Select a Template Template");
		chooser.setFileFilter(new GenericFileFilter("KawigiEdit Template Templates (*.ktt)", "ktt"));
		if (chooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION)
			templateLocation = chooser.getSelectedFile();
	}
}