package kawigi.config;
import javax.swing.*;
import javax.swing.event.*;
import java.util.*;
import kawigi.template.*;

/**
 *	A class that maps a TagLibrary to a JComboBox.  This is used by the editor to provide editing functionality.
 **/
public class TagLibEditorModel implements MutableComboBoxModel
{
	/**
	 *	The currently opened TagLibrary.
	 **/
	protected TagLibrary taglib;
	
	/**
	 *	The currently selected object.
	 **/
	private Object selected;
	
	/**
	 *	A list of ListDataListeners listening on this model.
	 **/
	private List listeners;
	
	/**
	 *	Creates a TagLibEditorModel instance mapping to the given TagLibrary.
	 **/
	public TagLibEditorModel(TagLibrary taglib)
	{
		this.taglib = taglib;
		listeners = new ArrayList();
	}
	
	/**
	 *	Returns the TagLibrary that is mapped from this editor model.
	 **/
	public TagLibrary getTagLibrary()
	{
		return taglib;
	}
	
	/**
	 *	Changes the TagLibrary represented by this TagLibEditorModel.
	 **/
	public void setTagLibrary(TagLibrary lib)
	{
		taglib = lib;
		fireEvent();
	}
	
	/**
	 *	Adds a new MethodInfo to the TagLibrary.
	 *	
	 *	<tt>obj</tt> should be an instance of MethodInfo, or else it won't be added.
	 **/
	public void addElement(Object obj)
	{
		if (obj instanceof MethodInfo)
		{
			taglib.addMethod((MethodInfo)obj);
			fireEvent();
		}
	}
	
	/**
	 *	Adds a new MethodInfo to the TagLibrary.
	 *	
	 *	It does this by calling addElement.  <tt>obj</tt> should be an instance of MethodInfo, or nothing
	 *	will happen.  <tt>index</tt> is completely disregarded - you don't get to choose where a new element
	 *	is added to the TagLibrary.  They are added by name into a TreeMap, so they are always inserted into
	 *	their lexicographical ordering.
	 **/
	public void insertElementAt(Object obj, int index)
	{
		addElement(obj);
	}
	
	/**
	 *	Removes a MethodInfo from the TagLibrary.
	 *	
	 *	Does nothing if <tt>obj</tt> is not a MethodInfo defined by the current TagLibrary.
	 **/
	public void removeElement(Object obj)
	{
		if (obj instanceof MethodInfo && taglib.defines(((MethodInfo)obj).getMethodName()))
		{
			taglib.remove((MethodInfo)obj);
			fireEvent();
		}
	}
	
	/**
	 *	Removes the indexth element from the TagLibrary, as mapped to the JComboBox.  Ignores requests to
	 *	remove element 0, because that is where the &lt;New Tag&gt; thing is.
	 **/
	public void removeElementAt(int index)
	{
		if (index > 0 && index < getSize())
		{
			taglib.remove(taglib.getMethods()[index-1]);
			fireEvent();
		}
	}
	
	/**
	 *	Returns the object designated as being selected.
	 **/
	public Object getSelectedItem()
	{
		return selected;
	}
	
	/**
	 *	Changes which object is designated as being selected.
	 **/
	public void setSelectedItem(Object anItem)
	{
		selected = anItem;
	}
	
	/**
	 *	Adds the ListDataListener to the list of listeners that will be notified of changes to this model.
	 **/
	public void addListDataListener(ListDataListener l)
	{
		listeners.add(l);
	}
	
	/**
	 *	Returns the element in the list at the given index.
	 *	
	 *	Returns the string, "&lt;New Tag&gt;" for index 0, otherwise returns the MethodInfo from the TagLibrary
	 *	at position <tt>index-1</tt>.
	 **/
	public Object getElementAt(int index)
	{
		if (index == 0)
			return "<New Tag>";
		MethodInfo[] methods = taglib.getMethods();
		return methods[index-1];
	}
	
	/**
	 *	Returns the number of elements to be rendered in the JComboBox, which is one more than the number of
	 *	Tags in the TagLibrary.
	 **/
	public int getSize()
	{
		return taglib.getSize()+1;
	}
	
	/**
	 *	Removes a ListDataListener so that it will no longer receive change events from this model.
	 **/
	public void removeListDataListener(ListDataListener l)
	{
		listeners.remove(l);
	}
	
	/**
	 *	Fires a CONTENTS_CHANGED event to all the listeners to this model.
	 **/
	protected void fireEvent()
	{
		for (int i=0; i<listeners.size(); i++)
			((ListDataListener)listeners.get(i)).contentsChanged(new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, 0, getSize()));
	}
}