package kawigi.config;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.undo.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import kawigi.editor.*;
import kawigi.template.*;
import kawigi.util.*;
import com.topcoder.client.contestApplet.common.LocalPreferences;

/**
 *	This class represents the tab in the config dialog for creating/editing tag libraries.
 **/
public class TagLibEditor extends JPanel implements KeyListener, UndoableEditListener, ActionListener, ItemListener
{
	private NoWrapJTextPane methodCodeEditor, callingCodeEditor;
	private JComboBox methodNameBox;
	private JButton saveButton, saveAsButton, openButton, deleteButton, addButton;
	private UndoManager methodUndo, callingUndo;
	private TagLibEditorModel model;
	private File saveFile;
	private TagNameEditor nameEditor;
	
	/**
	 *	Creates a new Panel with the necessary widgets on it.
	 **/
	public TagLibEditor()
	{
		setLayout(new BorderLayout());
		JPanel editorPanel = new JPanel();
		editorPanel.setLayout(new BoxLayout(editorPanel, BoxLayout.Y_AXIS));
		methodNameBox = new JComboBox(model = new TagLibEditorModel(new TagLibrary()));
		methodNameBox.setMaximumSize(new Dimension(methodNameBox.getMaximumSize().width, methodNameBox.getPreferredSize().height));
		methodNameBox.setEditable(true);
		methodNameBox.addItemListener(this);
		methodNameBox.setEditor(nameEditor = new TagNameEditor());
		JLabel nameLabel = new JLabel("Tag name:");
		nameLabel.setForeground(Color.white);
		JPanel namePanel = new JPanel();
		namePanel.setLayout(new BoxLayout(namePanel, BoxLayout.X_AXIS));
		namePanel.add(nameLabel);
		namePanel.add(methodNameBox);
		editorPanel.add(namePanel);
		editorPanel.add(makeLabel("Method code:"));
		methodCodeEditor = new NoWrapJTextPane();
		editorPanel.add(initializeJavaEditor(methodCodeEditor));
		editorPanel.add(makeLabel("Calling code:"));
		callingCodeEditor = new NoWrapJTextPane();
		editorPanel.add(initializeJavaEditor(callingCodeEditor));
		add(editorPanel);
		methodUndo = new UndoManager();
		callingUndo = new UndoManager();
		
		JPanel buttonPanel = new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
		saveButton = new JButton("Save");
		saveButton.addActionListener(this);
		buttonPanel.add(saveButton);
		saveAsButton = new JButton("Save As...");
		saveAsButton.addActionListener(this);
		buttonPanel.add(saveAsButton);
		openButton = new JButton("Open...");
		openButton.addActionListener(this);
		buttonPanel.add(openButton);
		deleteButton = new JButton("Delete");
		deleteButton.addActionListener(this);
		buttonPanel.add(deleteButton);
		addButton = new JButton("Add");
		addButton.addActionListener(this);
		buttonPanel.add(addButton);
		add(buttonPanel, BorderLayout.SOUTH);
	}
	
	/**
	 *	Does the common initialization of an editor pane.
	 *	
	 *	Eventually this stuff should all be encapsulated in its own class probably, so the basic functionality
	 *	of an "editor component" is consistent between parts of the application without the overhead that this
	 *	takes.
	 **/
	private JScrollPane initializeJavaEditor(NoWrapJTextPane editorPane)
	{
		editorPane.setContentType("text/Java");
		initializeTextComponent(editorPane);
		editorPane.addKeyListener(this);
		editorPane.getStyledDocument().addUndoableEditListener(this);
		editorPane.setDragEnabled(true);
		LineNumbers lines = new LineNumbers(editorPane);
		JViewport rowViewport = new JViewport();
		rowViewport.setView(lines);
		rowViewport.setBackground(editorPane.getBackground());
		JScrollPane scroll = new JScrollPane(editorPane, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		scroll.setRowHeader(rowViewport);
		JPanel corner = new JPanel();
		corner.setBackground(editorPane.getBackground());
		scroll.setCorner(JScrollPane.LOWER_LEFT_CORNER, corner);
		return scroll;
	}
	
	/**
	 *	More general initialization of any text component to go with the user settings.
	 **/
	private void initializeTextComponent(JTextComponent textThing)
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		textThing.setBackground(prefs.getColor("kawigi.editor.background"));
		textThing.setForeground(prefs.getColor("kawigi.editor.foreground"));
		textThing.setCaretColor(prefs.getColor("kawigi.editor.foreground"));
		textThing.setSelectionColor(prefs.getColor("kawigi.editor.SelectionColor"));
		textThing.setSelectedTextColor(prefs.getColor("kawigi.editor.SelectedTextColor"));
		textThing.setFont(new Font(prefs.getProperty("kawigi.editor.font"), Font.PLAIN, prefs.getFontSize("kawigi.editor.font.size")));
	}
	
	/**
	 *	Little utility JLabel factory that makes sure my JLabels display with white
	 *	text.
	 **/
	private JLabel makeLabel(String text)
	{
		JLabel label = new JLabel(text, SwingConstants.LEFT);
		label.setForeground(Color.white);
		label.setMaximumSize(new Dimension(Integer.MAX_VALUE, label.getMaximumSize().height));
		return label;
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	Specifically listens for ctrl+z and ctrl+y for undo and redo, respectively.
	 **/
	public void keyPressed(KeyEvent e)
	{
		if (e.getKeyCode() == KeyEvent.VK_Z && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
		{
			try
			{
				if (e.getSource() == methodCodeEditor)
					methodUndo.undo();
				else if (e.getSource() == callingCodeEditor)
					callingUndo.undo();
			}
			catch (CannotUndoException ex)
			{
			}

		}
		if (e.getKeyCode() == KeyEvent.VK_Y && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
		{
			try
			{
				if (e.getSource() == methodCodeEditor)
					methodUndo.redo();
				else if (e.getSource() == callingCodeEditor)
					callingUndo.redo();
			}
			catch (CannotRedoException ex)
			{
			}
		}
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 **/
	public void keyReleased(KeyEvent e)
	{
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	This handles the auto-indent.
	 **/
	public void keyTyped(KeyEvent e)
	{
		NoWrapJTextPane current = (NoWrapJTextPane)e.getSource();
		if (e.getKeyChar() == '\n')
		{
			String text = current.getText().replaceAll("\\r", "");
			String line = "";
			String indentation = "";
			int lines = 0;
			for (int ind = -1; ind < current.getCaretPosition(); ind = text.indexOf('\n', ind+1))
			{
				if (ind < 0 && lines != 0)
					break;
				lines++;
				indentation = (line.trim().length() == 0) ? line : line.substring(0, line.indexOf(line.trim()));
				line = (text.indexOf('\n', ind+1) < 0) ? text.substring(ind+1) : text.substring(ind+1, text.indexOf('\n', ind+1));
			}
			if ((e.getModifiersEx()&InputEvent.SHIFT_DOWN_MASK) != 0)
				indentation = "\n" + indentation;
			current.replaceSelection(indentation);
		}
	}
	
	/**
	 *	Stores away edits so they can be undone later.
	 **/
	public void undoableEditHappened(UndoableEditEvent e)
	{
		if (e.getSource() == methodCodeEditor)
			methodUndo.addEdit(e.getEdit());
		else if (e.getSource() == callingCodeEditor)
			callingUndo.addEdit(e.getEdit());
	}
	
	/**
	 *	Processes actions from the buttons.
	 **/
	public void actionPerformed(ActionEvent e)
	{
		if (e.getSource() == deleteButton)
		{
			model.removeElement(model.getSelectedItem());
		}
		else if (e.getSource() == addButton)
		{
			model.removeElement(model.getSelectedItem());
			MethodInfo method = new MethodInfo();
			String[] calling = callingCodeEditor.getText().split("[\\r\\n]+");
			String[] methodcode = methodCodeEditor.getText().split("[\\r\\n]+");
			method.setCallingCodeTemplate(calling);
			method.setMethodCode(methodcode);
			method.setMethodName(nameEditor.getText());
			model.addElement(method);
		}
		else
		{
			LocalPreferences prefs = LocalPreferences.getInstance();
			File localDir = new File(prefs.getProperty("kawigi.localpath"));
			if (!localDir.exists())
				localDir.mkdir();
			JFileChooser chooser = new JFileChooser(localDir);
			chooser.setFileFilter(new GenericFileFilter("KawigiEdit Tag Libraries (*.tlb)", "tlb"));
			if (e.getSource() == openButton)
			{
				if (chooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION)
				{
					saveFile = chooser.getSelectedFile();
					model.setTagLibrary(TagLibrary.read(saveFile));
				}
			}
			else if (e.getSource() == saveButton || e.getSource() == saveAsButton)
			{
				//first put the changes to the currently opened tag into the TagLibrary
				MethodInfo method;
				if (methodNameBox.getSelectedItem() instanceof MethodInfo)
					method = (MethodInfo)methodNameBox.getSelectedItem();
				else
					method = new MethodInfo();
				String[] calling = callingCodeEditor.getText().split("[\\r\\n]+");
				String[] methodcode = methodCodeEditor.getText().split("[\\r\\n]+");
				method.setCallingCodeTemplate(calling);
				method.setMethodCode(methodcode);
				if (!(methodNameBox.getSelectedItem() instanceof MethodInfo) && !nameEditor.getText().equals("<New Tag>"))
				{
					method.setMethodName(nameEditor.getText());
					model.addElement(method);
				}
				else if (methodNameBox.getSelectedItem() instanceof MethodInfo && !nameEditor.getText().equals(((MethodInfo)methodNameBox.getSelectedItem()).getMethodName()) && !model.getTagLibrary().defines(nameEditor.getText()))
				{
					model.getTagLibrary().remove((MethodInfo)methodNameBox.getSelectedItem());
					method.setMethodName(nameEditor.getText());
					model.addElement(method);
				}
				//sometimes bring up a dialog.
				if (e.getSource() == saveAsButton || saveFile == null)
				{
					if (chooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION)
					{
						saveFile = chooser.getSelectedFile();
						model.getTagLibrary().save(saveFile);
					}
				}
				else
					model.getTagLibrary().save(saveFile);
			}
		}
	}
	
	private boolean doingIt;
	
	/**
	 *	Processes changes in selection on the combo box.
	 **/
	public void itemStateChanged(ItemEvent e)
	{
		if (e.getStateChange() == ItemEvent.DESELECTED)
		{
			MethodInfo method;
			if (e.getItem() instanceof MethodInfo)
				method = (MethodInfo)e.getItem();
			else
				method = new MethodInfo();
			String[] calling = callingCodeEditor.getText().split("[\\r\\n]+");
			String[] methodcode = methodCodeEditor.getText().split("[\\r\\n]+");
			method.setCallingCodeTemplate(calling);
			method.setMethodCode(methodcode);
			if (!(e.getItem() instanceof MethodInfo) && !nameEditor.getText().equals("<New Tag>"))
			{
				method.setMethodName(nameEditor.getText());
				//Avoiding infinite event recursion.  For some reason, firing a ListDataEvent because of a different
				//element being added causes another deselected event to be fired.
				if (!doingIt)
				{
					doingIt = true;
					model.addElement(method);
					doingIt = false;
				}
			}
			else if (e.getItem() instanceof MethodInfo && !nameEditor.getText().equals(((MethodInfo)e.getItem()).getMethodName()) && !model.getTagLibrary().defines(nameEditor.getText()))
			{
				model.getTagLibrary().remove((MethodInfo)e.getItem());
				method.setMethodName(nameEditor.getText());
				model.addElement(method);
			}
		}
		else if (e.getStateChange() == ItemEvent.SELECTED)
		{
			if (e.getItem() instanceof MethodInfo)
			{
				MethodInfo method = (MethodInfo)e.getItem();
				String[] calling = method.getCallingCodeTemplate();
				String[] methodcode = method.getMethodCode();
				String callingString = "";
				for (int i=0; i<calling.length; i++)
					callingString += calling[i] + "\n";
				String methodString = "";
				for (int i=0; i<methodcode.length; i++)
					methodString += methodcode[i] + "\n";
				callingCodeEditor.setText(callingString);
				methodCodeEditor.setText(methodString);
			}
			else
			{
				callingCodeEditor.setText("");
				methodCodeEditor.setText("");
			}
		}
	}
}