package kawigi;
import kawigi.config.*;
import kawigi.challenge.*;
import kawigi.snippet.*;
import kawigi.util.*;
import kawigi.editor.*;
import kawigi.timer.*;
import java.lang.reflect.*;
import java.io.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.undo.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import com.topcoder.shared.language.*;
import com.topcoder.shared.problem.*;
import com.topcoder.client.contestant.*;
import com.topcoder.client.contestApplet.common.LocalPreferences;
import kawigi.language.*;

/**
 *	KawigiEdit - an open-source, fully-featured editor plugin for the Top Coder Arena.  It was
 *		written by me for me, so don't be surprised if the features aren't in line with what you
 *		want.  If you want something that's not here, talk to me, and either I will add it, or
 *		if I don't care, you can.
 **/
public class KawigiEdit implements ActionListener, CaretListener, KeyListener, MouseListener, UndoableEditListener, FocusListener, DocumentListener
{
	/**
	 *	The current version of KawigiEdit.
	 **/
	public static String VERSION = "1.1";
	protected NoWrapJTextPane textArea, localCodePane, currentPopupThing;
	protected JTextArea testingPane;
	protected JTabbedPane tabs;
	protected JPanel mainPanel, snippetPanel, replacePanel, varReplacePanel;
	protected JFileChooser localFileChooser;
	private ProblemComponent component;
	private JButton generateButton, testButton, saveButton, killButton, snippetButton, replaceButton, varReplaceButton, challengeToolsButton, configButton, openButton, loadButton;
	private JLabel lineLabel;
	private File tempdir;
	private ProcessContainer proc;
	private JPopupMenu popup;
	private JCheckBox useRegexBox;
	private Category baseCategory;
	private JTextField nameField, categoryField, replaceField, replaceWithField, varReplaceField, varReplaceWithField;
	private String codeSelection;
	private UndoManager undo;
	private ExtendedLanguage language;
	private boolean useStdUI;
	private ProblemTimer timer;
	
	/**
	 *	Initializes KawigiEdit and checks to make sure certain settings are available.
	 **/
	public KawigiEdit()
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		//This is how you figure out what settings you can get from the applet:
		/*Enumeration it = prefs.getKeys();
		while (it.hasMoreElements())
		{
			String key = (String)it.nextElement();
			System.out.println(key + "=" + prefs.getProperty(key));
		}*/
		if (prefs.getProperty("kawigi.stdui") != null)
			useStdUI = prefs.isTrue("kawigi.stdui");
		else
			prefs.setTrue("kawigi.stdui", useStdUI = false);
		if (prefs.getProperty("kawigi.localpath") != null)
			tempdir = new File(prefs.getProperty("kawigi.localpath"));
		else
		{
			tempdir = new File("testprograms");
			prefs.setProperty("kawigi.localpath", tempdir.getPath());
		}
		if (!tempdir.exists())
			tempdir.mkdir();
		textArea = new NoWrapJTextPane();
		if (prefs.getProperty("kawigi.editor.background") != null)
			textArea.setBackground(prefs.getColor("kawigi.editor.background"));
		else
		{
			textArea.setBackground(Color.black);
			prefs.setColor("kawigi.editor.background", Color.black);
		}
		if (prefs.getProperty("kawigi.editor.foreground") != null)
		{
			textArea.setForeground(prefs.getColor("kawigi.editor.foreground"));
			textArea.setCaretColor(prefs.getColor("kawigi.editor.foreground"));
		}
		else
		{
			textArea.setForeground(Color.white);
			textArea.setCaretColor(Color.white);
			prefs.setColor("kawigi.editor.foreground", Color.white);
		}
		String fontface;
		if (prefs.getProperty("kawigi.editor.font") != null)
			fontface = prefs.getProperty("kawigi.editor.font");
		else
			prefs.setProperty("kawigi.editor.font", fontface = "Monospaced");
		int fontsize;
		if (prefs.getProperty("kawigi.editor.font.size") != null)
			fontsize = prefs.getFontSize("kawigi.editor.font.size");
		else
			prefs.setFontSize("kawigi.editor.font.size", Integer.toString(fontsize = 12));
		textArea.setFont(new Font(fontface, Font.PLAIN, fontsize));
		textArea.addCaretListener(this);
		textArea.addKeyListener(this);
		textArea.addMouseListener(this);
		textArea.getStyledDocument().addUndoableEditListener(this);
		textArea.getStyledDocument().addDocumentListener(this);
		textArea.setDragEnabled(true);
		if (prefs.getProperty("kawigi.editor.SelectionColor") != null)
			textArea.setSelectionColor(prefs.getColor("kawigi.editor.SelectionColor"));
		else
			prefs.setColor("kawigi.editor.SelectionColor", textArea.getSelectionColor());
		if (prefs.getProperty("kawigi.editor.SelectedTextColor") != null)
			textArea.setSelectedTextColor(prefs.getColor("kawigi.editor.SelectedTextColor"));
		else
			prefs.setColor("kawigi.editor.SelectedTextColor", textArea.getSelectedTextColor());
		LineNumbers lines = new LineNumbers(textArea);
		JViewport rowViewport = new JViewport();
		rowViewport.setView(lines);
		rowViewport.setBackground(textArea.getBackground());
		tabs = new JTabbedPane();
		tabs.addFocusListener(this);
		JScrollPane scrollpane1 = new JScrollPane(textArea, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		scrollpane1.setRowHeader(rowViewport);
		JPanel corner = new JPanel();
		corner.setBackground(textArea.getBackground());
		scrollpane1.setCorner(JScrollPane.LOWER_LEFT_CORNER, corner);
		tabs.add(scrollpane1, "Source Code");
		
		if (prefs.getProperty("kawigi.testing.font") != null)
			fontface = prefs.getProperty("kawigi.testing.font");
		else
			prefs.setProperty("kawigi.testing.font", fontface = "Monospaced");
		if (prefs.getProperty("kawigi.editor.font.size") != null)
			fontsize = prefs.getFontSize("kawigi.testing.font.size");
		else
			prefs.setFontSize("kawigi.testing.font.size", Integer.toString(fontsize = 12));
		testingPane = new JTextArea();
		testingPane.setText("No results yet");
		if (prefs.getProperty("kawigi.testing.background") != null)
			testingPane.setBackground(prefs.getColor("kawigi.testing.background"));
		else
		{
			testingPane.setBackground(Color.white);
			prefs.setColor("kawigi.testing.background", Color.white);
		}
		if (prefs.getProperty("kawigi.testing.foreground") != null)
		{
			testingPane.setForeground(prefs.getColor("kawigi.testing.foreground"));
			testingPane.setCaretColor(prefs.getColor("kawigi.testing.foreground"));
		}
		else
		{
			testingPane.setForeground(Color.black);
			testingPane.setCaretColor(Color.black);
			prefs.setColor("kawigi.testing.foreground", Color.black);
		}
		testingPane.setEditable(false);
		testingPane.setFont(new Font(fontface, Font.PLAIN, fontsize));
		tabs.add(new JScrollPane(testingPane, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED), "Test Results");
		
		localFileChooser = new JFileChooser(tempdir);
		localFileChooser.addActionListener(this);
		
		localCodePane = new NoWrapJTextPane();
		localCodePane.setBackground(textArea.getBackground());
		localCodePane.setForeground(textArea.getForeground());
		localCodePane.setCaretColor(textArea.getCaretColor());
		localCodePane.setFont(textArea.getFont());
		localCodePane.setEditable(false);
		localCodePane.addMouseListener(this);
		lines = new LineNumbers(localCodePane);
		rowViewport = new JViewport();
		rowViewport.setView(lines);
		rowViewport.setBackground(textArea.getBackground());
		JScrollPane scrollpane2 = new JScrollPane(localCodePane, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		scrollpane2.setRowHeader(rowViewport);
		corner = new JPanel();
		corner.setBackground(textArea.getBackground());
		scrollpane2.setCorner(JScrollPane.LOWER_LEFT_CORNER, corner);
		openButton = new JButton("Open...");
		openButton.addActionListener(this);
		JPanel openPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		openPanel.add(openButton);
		
		JPanel localCodePanel = new JPanel(new BorderLayout());
		localCodePanel.add(openPanel, BorderLayout.SOUTH);
		localCodePanel.add(scrollpane2);
		
		tabs.add(localCodePanel, "Local Code");
		
		mainPanel = new JPanel(new BorderLayout());
		mainPanel.add(tabs, BorderLayout.CENTER);
		
		generateButton = new JButton("Generate code");
		generateButton.addActionListener(this);
		JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		buttonPanel.add(generateButton);
		
		challengeToolsButton = new JButton("Challenge tools");
		challengeToolsButton.addActionListener(this);
		challengeToolsButton.setMnemonic(KeyEvent.VK_H);
		challengeToolsButton.setDisplayedMnemonicIndex(1);
		buttonPanel.add(challengeToolsButton);
		
		configButton = new JButton("Config");
		configButton.addActionListener(this);
		buttonPanel.add(configButton);
		
		testButton = new JButton("Run Tests");
		testButton.addActionListener(this);
		testButton.setMnemonic(KeyEvent.VK_R);
		testButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(testButton);
		
		saveButton = new JButton("Save");
		saveButton.addActionListener(this);
		saveButton.setMnemonic(KeyEvent.VK_A);
		saveButton.setDisplayedMnemonicIndex(1);
		buttonPanel.add(saveButton);
		
		loadButton = new JButton("Load");
		loadButton.addActionListener(this);
		loadButton.setMnemonic(KeyEvent.VK_L);
		loadButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(loadButton);
		
		killButton = new JButton("Kill Process");
		killButton.addActionListener(this);
		killButton.setMnemonic(KeyEvent.VK_K);
		killButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(killButton);
		
		lineLabel = new JLabel("Line: 0");
		lineLabel.setForeground(Color.white);
		lineLabel.setFont(new Font("Monospaced", Font.BOLD, 14));
		buttonPanel.add(lineLabel);
		
		mainPanel.add(buttonPanel, BorderLayout.SOUTH);
		
		if (prefs.getProperty("kawigi.timer.show") == null)
			prefs.setTrue("kawigi.timer.show", true);
		if (prefs.isTrue("kawigi.timer.show"))
		{
			timer = new ProblemTimer();
			mainPanel.add(timer, BorderLayout.WEST);
		}
		
		File f = new File(tempdir, "snippets.dat");
		if (f.exists())
		{
			try
			{
				ObjectInputStream in = new ObjectInputStream(new BufferedInputStream(new FileInputStream(f)));
				baseCategory = (Category)in.readObject();
				in.close();
			}
			catch (IOException ex)
			{
			}
			catch (ClassNotFoundException ex)
			{
			}
		}
		if (baseCategory == null)
			baseCategory = new Category("");
		baseCategory.propagateActionListener(this);
		
		nameField = new JTextField();
		nameField.addActionListener(this);
		categoryField = new JTextField(12);
		categoryField.addActionListener(this);
		snippetButton = new JButton("Record Snippet");
		snippetButton.addActionListener(this);
		snippetPanel = new JPanel(new GridLayout(3, 2));
		JLabel temp = new JLabel("Snippet name:");
		temp.setForeground(Color.white);
		snippetPanel.add(temp);
		snippetPanel.add(nameField);
		temp = new JLabel("Category:");
		temp.setForeground(Color.white);
		snippetPanel.add(temp);
		snippetPanel.add(categoryField);
		snippetPanel.add(new JLabel());
		snippetPanel.add(snippetButton);
		
		replaceField = new JTextField();
		replaceField.addActionListener(this);
		replaceWithField = new JTextField();
		replaceWithField.addActionListener(this);
		replaceButton = new JButton("Replace");
		replaceButton.addActionListener(this);
		replacePanel = new JPanel(new GridLayout(3, 2));
		temp = new JLabel("Replace:");
		temp.setForeground(Color.white);
		replacePanel.add(temp);
		replacePanel.add(replaceField);
		temp = new JLabel("With:");
		temp.setForeground(Color.white);
		replacePanel.add(temp);
		replacePanel.add(replaceWithField);
		useRegexBox = new JCheckBox("regex");
		if (prefs.getProperty("kawigi.editor.useregex") == null)
			prefs.setTrue("kawigi.editor.useregex", true);
		useRegexBox.setSelected(prefs.isTrue("kawigi.editor.useregex"));
		useRegexBox.addActionListener(this);
		useRegexBox.setBackground(Color.darkGray);
		useRegexBox.setForeground(Color.white);
		replacePanel.add(useRegexBox);
		replacePanel.add(replaceButton);
		
		varReplaceField = new JTextField();
		varReplaceField.addActionListener(this);
		varReplaceWithField = new JTextField();
		varReplaceWithField.addActionListener(this);
		varReplaceButton = new JButton("Change");
		varReplaceButton.addActionListener(this);
		varReplacePanel = new JPanel(new GridLayout(3, 2));
		temp = new JLabel("Change:");
		temp.setForeground(Color.white);
		varReplacePanel.add(temp);
		varReplacePanel.add(varReplaceField);
		temp = new JLabel("To:");
		temp.setForeground(Color.white);
		varReplacePanel.add(temp);
		varReplacePanel.add(varReplaceWithField);
		varReplacePanel.add(new JLabel());
		varReplacePanel.add(varReplaceButton);
		undo = new UndoManager();
	}
	
	/**
	 *	Saves the snippet database to disk in a file called "snippets.dat".
	 **/
	protected void saveSnippets()
	{
		try
		{
			ObjectOutputStream out = new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(new File(tempdir, "snippets.dat"))));
			out.writeObject(baseCategory);
			out.close();
		}
		catch (IOException ex)
		{
			System.err.println("Couldn't write snippets");
		}
	}
	
	/**
	 *	Handles events of all the buttons, menus, and so forth.
	 **/
	public void actionPerformed(ActionEvent e)
	{
		if (e.getSource() == useRegexBox)
		{
			LocalPreferences.getInstance().setTrue("kawigi.editor.useregex", useRegexBox.isSelected());
		}
		if (e.getSource() == generateButton)
		{
			if (component != null && language != null)
			{
				Skeleton code = language.getSkeleton(component);
				textArea.setText(code.getSource());
				textArea.setCaretPosition(code.getCaret());
			}
		}
		else if (e.getSource() == testButton)
		{
			testingPane.setText("");
			tabs.setSelectedIndex(tabs.indexOfTab("Test Results"));
			try
			{
				saveLocal();
				if (compileLocal())
				{
					testingPane.setText(testingPane.getText() + "Compiled\n\n");
					
					if (proc == null || proc.isDone())
					{
						Process p = Runtime.getRuntime().exec(language.getRunCommand(component, tempdir), null, tempdir);
						proc = new ProcessContainer(p, testingPane);
						proc.start();
					}
					else
					{
						testingPane.setText(testingPane.getText() + "Error: Can't start new process while another is running.\n");
					}
				}
				else
					testingPane.setText(testingPane.getText() + "Compiling errors\n");
			}
			catch (Throwable t)
			{
				testingPane.setText(testingPane.getText() + "My error: " + t + "\n");
				t.printStackTrace();
			}
		}
		else if (e.getSource() == saveButton)
		{
			try
			{
				saveLocal();
			}
			catch (Throwable t)
			{
				lineLabel.setText("Couldn't save!");
			}
		}
		else if (e.getSource() == loadButton)
		{
			try
			{
				if (!tempdir.exists())
					tempdir.mkdir();
				BufferedReader inFile = new BufferedReader(new FileReader(new File(tempdir, language.getFileName(component))));
				String text = "";
				String line;
				while ((line = inFile.readLine()) != null)
					text += line + "\n";
				textArea.setText(text);
			}
			catch (Throwable t)
			{
				lineLabel.setText("Couldn't load file.");
			}
		}
		else if (e.getSource() == killButton)
		{
			if (proc != null &! proc.isDone())
				proc.kill();
		}
		else if (e.getSource() == localFileChooser)
		{
		}
		else if (e.getSource() == openButton)
		{
			if (localFileChooser.showOpenDialog(null) == JFileChooser.APPROVE_OPTION)
			{
				File f = localFileChooser.getSelectedFile();
				try
				{
					BufferedReader inFile = new BufferedReader(new FileReader(f));
					String text = "";
					String line;
					while ((line = inFile.readLine()) != null)
						text += line + "\n";
					inFile.close();
					if (f.getName().toLowerCase().endsWith(".cs"))
						localCodePane.setContentType("text/" + CSharpLanguage.CSHARP_LANGUAGE.getName());
					else if (f.getName().toLowerCase().endsWith(".java"))
						localCodePane.setContentType("text/" + JavaLanguage.JAVA_LANGUAGE.getName());
					else if (f.getName().toLowerCase().endsWith(".vb"))
						localCodePane.setContentType("text/" + VBLanguage.VB_LANGUAGE.getName());
					else
						localCodePane.setContentType("text/" + CPPLanguage.CPP_LANGUAGE.getName());
					localCodePane.setText(text);
					((JViewport)localCodePane.getParent()).setViewPosition(new Point(0, 0));
				}
				catch (IOException ex)
				{
					localCodePane.setText("IOException thrown!");
				}
			}
		}
		else if (e.getSource() == snippetButton || e.getSource() == nameField || e.getSource() == categoryField)
		{
			Snippet s = new Snippet(codeSelection, nameField.getText());
			baseCategory.add(s, categoryField.getText());
			s.addActionListener(this);
			if (popup != null)
				popup.setVisible(false);
			saveSnippets();
		}
		else if (e.getSource() == replaceButton || e.getSource() == replaceField || e.getSource() == replaceWithField)
		{
			if (replaceField.getText().length() > 0 && replaceWithField.getText().length() > 0)
			{
				if (textArea.getSelectedText() == null || textArea.getSelectedText().length() == 0)
				{
					if (useRegexBox.isSelected())
						textArea.setText(textArea.getText().replaceAll(replaceField.getText(), replaceWithField.getText()));
					else
					{
						String text = textArea.getText();
						String replace = replaceField.getText();
						String replaceWith = replaceWithField.getText();
						int ind = 0;
						while ((ind = text.indexOf(replace, ind)) > 0)
						{
							text = text.substring(0, ind) + replaceWith + text.substring(ind + replace.length());
							ind += replaceWith.length();
						}
						textArea.setText(text);
					}
				}
				else
				{
					if (useRegexBox.isSelected())
						textArea.replaceSelection(textArea.getSelectedText().replaceAll(replaceField.getText(), replaceWithField.getText()));
					else
					{
						String text = textArea.getSelectedText();
						String replace = replaceField.getText();
						String replaceWith = replaceWithField.getText();
						int ind = 0;
						while ((ind = text.indexOf(replace, ind)) > 0)
						{
							text = text.substring(0, ind) + replaceWith + text.substring(ind + replace.length());
							ind += replaceWith.length();
						}
						textArea.replaceSelection(text);
					}
				}
				if (popup != null)
					popup.setVisible(false);
			}
		}
		else if (e.getSource() == varReplaceButton || e.getSource() == varReplaceField || e.getSource() == varReplaceWithField)
		{
			if (varReplaceField.getText().length() > 0 && varReplaceWithField.getText().length() > 0)
			{
				if (textArea.getSelectedText() == null)
					textArea.setText(textArea.getText().replaceAll("([^a-zA-Z0-9_])" + varReplaceField.getText() + "([^a-zA-Z0-9_])", "$1" + varReplaceWithField.getText() + "$2"));
				else
					textArea.replaceSelection(textArea.getSelectedText().replaceAll("([^a-zA-Z0-9_])" + varReplaceField.getText() + "([^a-zA-Z0-9_])", "$1" + varReplaceWithField.getText() + "$2"));
				if (popup != null)
					popup.setVisible(false);
			}
		}
		else if (e.getSource() == challengeToolsButton)
		{
			HistoryLookup.getFrame().show();
		}
		else if (e.getSource() == configButton)
		{
			configure();
		}
		else if (e.getSource() instanceof Snippet)
		{
			if ((e.getModifiers()&ActionEvent.META_MASK) != 0)
				((Snippet)e.getSource()).delete();
			else if (currentPopupThing != null)
				currentPopupThing.replaceSelection(e.getSource().toString());
		}
	}
	
	/**
	 *	Saves the current file to disk in the configured local directory with the appropriate name.
	 **/
	protected void saveLocal() throws IOException
	{
		if (!tempdir.exists())
			tempdir.mkdir();
		PrintWriter outFile = new PrintWriter(new FileWriter(new File(tempdir, language.getFileName(component))));
		outFile.println(textArea.getText());
		if (language != null && language.getId() == CPPLanguage.ID)
		{
			outFile.println("int main() {");
			outFile.println("int time;");
			outFile.println("\tbool errors = false;");
			outFile.println("\t");
			for (int i=0; i<component.getTestCases().length; i++)
			{
				outFile.println("\ttime = test" + i + "();");
				outFile.println("\tif (time < 0)");
				outFile.println("\t\terrors = true;");
				outFile.println("\t");
			}
			outFile.println("\tif (!errors)");
			outFile.println("\t\tcout <<\"You\'re a stud (at least on the example cases)!\" <<endl;");
			outFile.println("\telse");
			outFile.println("\t\tcout <<\"Some of the test cases had errors.\" <<endl;");
			outFile.println("}");
		}
		outFile.close();
	}
	
	/**
	 *	Compiles the saved file locally.
	 **/
	protected boolean compileLocal() throws Exception
	{
		if (proc == null || proc.isDone())
		{
			Process p = Runtime.getRuntime().exec(language.getCompileCommand(component), null, tempdir);
			proc = new ProcessContainer(p, testingPane, false);
			proc.start();
			p.waitFor();
			return proc.endVal() == 0;
		}
		else
		{
			testingPane.setText(testingPane.getText() + "Error: Can't compile while another process is running\n");
			return false;
		}
	}
	
	/**
	 *	Returns the magic KawigiEdit panel. 
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public JPanel getEditorPanel()
	{
		return mainPanel;
	}
	
	/**
	 *	Returns the text in the editor.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public String getSource()
	{
		return textArea.getText();
	}
	
	/**
	 *	Sets the text in the editor.
	 *	
	 *	This implementation will ignore the request if the source provided is empty.
	 *	This is to maintain auto-generated code.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void setSource(String source)
	{
		if (source.length() > 0)
		{
			textArea.setText(source);
			((JViewport)textArea.getParent()).setViewPosition(new Point(0, 0));
		}
	}
	
	/**
	 *	Empties the text pane.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void clear()
	{
		textArea.setText("");
	}
	
	/**
	 *	Enables/disables the text pane.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void setTextEnabled(Boolean b)
	{
		textArea.setEnabled(b.booleanValue());
	}
	
	/**
	 *	Notifies the editor of a new problem being opened, or the language being changed, or whatever.
	 *	
	 *	This resets some of the state, including the undo manager.  If the editor has no text yet, this
	 *	method will also generate skeleton and test code for the problem.  Eventually in the right language, even.
	 **/
	public void setProblemComponent(ProblemComponentModel component, Language lang, com.topcoder.shared.problem.Renderer renderer)
	{
		language = ExtendedLanguageFactory.getLanguage(lang);
		textArea.setContentType("text/" + language.getName());
		textArea.getStyledDocument().addUndoableEditListener(this);
		undo.discardAllEdits();
		this.component = component.getComponent();
		if (timer != null)
			timer.select(this.component.getProblemId(), component.getPoints().doubleValue());
		tabs.setSelectedIndex(tabs.indexOfTab("Source Code"));
		if (textArea.getText().length() == 0)
		{
			Skeleton code = language.getSkeleton(this.component);
			textArea.setText(code.getSource());
			textArea.setCaretPosition(code.getCaret());
			((JViewport)textArea.getParent()).setViewPosition(new Point(0, 0));
		}
	}
	
	/**
	 *	Clears the text pane for a new problem or something.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void startUsing()
	{
		textArea.setText("");
	}
	
	/**
	 *	Doesn't do anything.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void stopUsing()
	{
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	A lot of keystrokes were added here for version 1.1 due to requests from various places.
	 **/
	public void keyPressed(KeyEvent e)
	{
		//ctrl+z and ctrl+y are mine, ctrl+shift+z is something I got from Kate (the KDE text editor),
		//alt+backspace is for dplass, shift+alt+backspace I found out about from the VS keystroke documentation
		if ((e.getKeyCode() == KeyEvent.VK_Z && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0 && (e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) == 0)
			|| (e.getKeyCode() == KeyEvent.VK_BACK_SPACE && (e.getModifiersEx()&InputEvent.ALT_DOWN_MASK) != 0 && (e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) == 0))
		{
			try
			{
				undo.undo();
			}
			catch (CannotUndoException ex)
			{
			}

		}
		else if ((e.getKeyCode() == KeyEvent.VK_Y && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
			|| (e.getKeyCode() == KeyEvent.VK_Z && (e.getModifiersEx() & (InputEvent.CTRL_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK)) == (InputEvent.CTRL_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK))
			|| (e.getKeyCode() == KeyEvent.VK_BACK_SPACE && (e.getModifiersEx() & (InputEvent.ALT_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK)) == (InputEvent.ALT_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK)))
		{
			try
			{
				undo.redo();
			}
			catch (CannotRedoException ex)
			{
			}
		}
		//These three are for NeverMore, apparently they're from his use of some old Borland IDE,
		//and they're supported by a number of other editors/word processors:
		else if (e.getKeyCode() == KeyEvent.VK_INSERT && (e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0)
			textArea.copy();
		else if (e.getKeyCode() == KeyEvent.VK_DELETE && (e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) != 0)
			textArea.cut();
		else if (e.getKeyCode() == KeyEvent.VK_INSERT && (e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) != 0)
			textArea.paste();
		//tmyklebu gave me this idea.  ctrl+left and ctrl+right apparently move the cursor left and right
		//a word by default, which I suppose is good enough, so I won't frob with them (in case someone
		//is actually used to that behavior anyways):
		//ctrl + up scrolls up (without moving the cursor)
		else if (e.getKeyCode() == KeyEvent.VK_UP && (e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0)
		{
			Point loc = ((JViewport)textArea.getParent()).getViewPosition();
			loc.y = Math.max(0, loc.y - textArea.getScrollableUnitIncrement(textArea.getParent().getBounds(), SwingConstants.VERTICAL, -1));
			((JViewport)textArea.getParent()).setViewPosition(loc);
			e.consume();
///			textArea.requestFocus();
		}
		//ctrl + down scrolls down (without moving the cursor)
		else if (e.getKeyCode() == KeyEvent.VK_DOWN && (e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0)
		{
			Point loc = ((JViewport)textArea.getParent()).getViewPosition();
			loc.y = Math.min(textArea.getSize().height-textArea.getParent().getSize().height, loc.y + textArea.getScrollableUnitIncrement(textArea.getParent().getBounds(), SwingConstants.VERTICAL, -1));
			((JViewport)textArea.getParent()).setViewPosition(loc);
			e.consume();
	///		textArea.requestFocus();
		}
		//another few for dplass:
		//ctrl + backspace = delete last word
		else if (e.getKeyCode() == KeyEvent.VK_BACK_SPACE && (e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0)
		{
			int currentIndex = textArea.getCaretPosition();
			String text = textArea.getText().replaceAll("\\r", "");
			boolean foundWord = false;
			int start = currentIndex-1;
			if (start >= 0 && text.charAt(start) != '\n')
			{
				while (start > 0 && Character.isWhitespace(text.charAt(start)) && text.charAt(start) != '\n')
					start --;
				if (start > 0 && (Character.isLetterOrDigit(text.charAt(start)) || text.charAt(start) == '_'))
				{
					do
					{
						start--;
					}
					while (start >= 0 && (Character.isLetterOrDigit(text.charAt(start)) || text.charAt(start) == '_'));
				}
				else if (start >= 0 && text.charAt(start) != '\n')
				{
					do
					{
						start--;
					}
					while (start >= 0 && !(Character.isLetterOrDigit(text.charAt(start)) || text.charAt(start) == '_') && !Character.isWhitespace(text.charAt(start)));
				}
			}
			else if (start >= 0)
				start--;
			
			textArea.setText(text.substring(0, start+1) + text.substring(currentIndex));
			textArea.setCaretPosition(start+1);
		}
		//ctrl + delete = delete next word
		else if (e.getKeyCode() == KeyEvent.VK_DELETE && (e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0)
		{
			int currentIndex = textArea.getCaretPosition();
			String text = textArea.getText().replaceAll("\\r", "");
			boolean foundWord = false;
			int start = currentIndex;
			while (start < text.length() && Character.isWhitespace(text.charAt(start)) && text.charAt(start) != '\n')
				start ++;
			if (start < text.length() && (Character.isLetterOrDigit(text.charAt(start)) || text.charAt(start) == '_'))
			{
				do
				{
					start++;
				}
				while (start < text.length() && (Character.isLetterOrDigit(text.charAt(start)) || text.charAt(start) == '_'));
			}
			else if (start < text.length() && text.charAt(start) != '\n')
			{
				do
				{
					start++;
				}
				while (start < text.length() && !(Character.isLetterOrDigit(text.charAt(start)) || text.charAt(start) == '_') && !Character.isWhitespace(text.charAt(start)));
			}
			else if (start < text.length())
				start++;
			
			textArea.setText(text.substring(0, currentIndex) + text.substring(start));
			textArea.setCaretPosition(currentIndex);
		}
		//ctrl + page down/page up switches tabs.  Of course, it only works if you were on the editor before, and you can't go back :-p
		else if (e.getKeyCode() == KeyEvent.VK_PAGE_DOWN && (e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0)
			tabs.setSelectedIndex((tabs.getSelectedIndex() + 1)%tabs.getTabCount());
		else if (e.getKeyCode() == KeyEvent.VK_PAGE_UP && (e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0)
			tabs.setSelectedIndex((tabs.getTabCount() + tabs.getSelectedIndex() - 1)%tabs.getTabCount());
		//selecting text and hitting tab indents all lines in the selection!  Woo-hoo!
		else if (e.getKeyCode() == KeyEvent.VK_TAB && (e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) == 0 && textArea.getSelectionStart() != textArea.getSelectionEnd())
		{
			StyledDocument doc = textArea.getStyledDocument();
			javax.swing.text.Element root = doc.getRootElements()[0];
			int startLine = root.getElementIndex(textArea.getSelectionStart());
			int endLine = root.getElementIndex(textArea.getSelectionEnd());
			String text = textArea.getText().replaceAll("\\r", "");
			for (int i=endLine; i>= startLine; i--)
			{
				javax.swing.text.Element line = root.getElement(i);
				text = text.substring(0, line.getStartOffset()) + "\t" + text.substring(line.getStartOffset());
			}
			textArea.setText(text);
			textArea.setCaretPosition(root.getElement(endLine).getEndOffset()-1);
			textArea.setSelectionStart(root.getElement(startLine).getStartOffset());
			e.consume();
		}
		//selecting text and hitting shift-tab unindents the text.  A little harder, but not bad.
		else if (e.getKeyCode() == KeyEvent.VK_TAB && (e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) != 0 && textArea.getSelectionStart() != textArea.getSelectionEnd())
		{
			StyledDocument doc = textArea.getStyledDocument();
			javax.swing.text.Element root = doc.getRootElements()[0];
			int startLine = root.getElementIndex(textArea.getSelectionStart());
			int endLine = root.getElementIndex(textArea.getSelectionEnd());
			String text = textArea.getText().replaceAll("\\r", "");
			int tabwidth = LocalPreferences.getInstance().getFontSize("kawigi.editor.tabstop");
			for (int i=endLine; i>= startLine; i--)
			{
				javax.swing.text.Element line = root.getElement(i);
				if (text.charAt(line.getStartOffset()) == '\t')
					text = text.substring(0, line.getStartOffset()) + text.substring(line.getStartOffset()+1);
				else if (text.charAt(line.getStartOffset()) == ' ')
				{
					int start = line.getStartOffset();
					int end;
					for (end = start; end < start + tabwidth && text.charAt(end) == ' '; end++)
						;
					if (end < start + tabwidth && text.charAt(end) == '\t')
						end ++;
					text = text.substring(0, start) + text.substring(end);
				}
			}
			textArea.setText(text);
			textArea.setCaretPosition(root.getElement(endLine).getEndOffset()-1);
			textArea.setSelectionStart(root.getElement(startLine).getStartOffset());
		}
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	Specifically listens for ctrl+i to popup the snippet popup menu.
	 **/
	public void keyReleased(KeyEvent e)
	{
		if (e.getKeyCode() == KeyEvent.VK_I && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
		{
			JPopupMenu popup = getPopup(textArea);
			Point pt = textArea.getCaret().getMagicCaretPosition();
			if (pt == null)
				pt = new Point();
			popup.show(textArea, pt.x, pt.y);
		}
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	This handles the auto-indent.
	 **/
	public void keyTyped(KeyEvent e)
	{
		if (e.getKeyChar() == '\n')
		{
			String text = textArea.getText().replaceAll("\\r", "");
			String line = "";
			String indentation = "";
			int lines = 0;
			for (int ind = -1; ind < textArea.getCaretPosition(); ind = text.indexOf('\n', ind+1))
			{
				if (ind < 0 && lines != 0)
					break;
				lines++;
				indentation = (line.trim().length() == 0) ? line : line.substring(0, line.indexOf(line.trim()));
				line = (text.indexOf('\n', ind+1) < 0) ? text.substring(ind+1) : text.substring(ind+1, text.indexOf('\n', ind+1));
			}
			//for some reason, shift+enter doesn't also do a return, so I just make it finish the job
			if ((e.getModifiersEx()&InputEvent.SHIFT_DOWN_MASK) != 0)
				indentation = "\n" + indentation;
			textArea.replaceSelection(indentation);
		}
		textArea.repaint();
	}
	
	/**
	 *	Listens for changes in caret position on the text pane.
	 *	
	 *	Updates the line number displayed on the bottom-right.
	 **/
	public void caretUpdate(CaretEvent e)
	{
		int lines = 0;
		String text = textArea.getText().replaceAll("\\r", "");
		int caret = e.getDot();
		for (int ind = -1; ind<caret; ind = text.indexOf('\n', ind+1))
		{
			if (ind < 0 && lines != 0)
				break;
			lines++;
		}
		lineLabel.setText("Line: " + lines);
	}
	
	/**
	 *	Brings up a configure dialog to set options in the editor plugin.
	 *	
	 *	If I were to pick just one thing to complain about in the plugin specification,
	 *	it would be the description on how to do this.  I found it completely vague,
	 *	and after reading still wasn't sure what it was supposed to do.  Originally,
	 *	this method actually froze or something, or otherwise didn't work, until I
	 *	made the configure dialog modal.  On the side, we are encouraged to use TopCoder's
	 *	preferences class, but it is completely undocumented and the package it is
	 *	found in is not specified.  So I hacked up an old program I wrote that reads
	 *	a jar and uses the jar classes as well as reflection classes to chart the
	 *	classes in the jar on a JTree, and looked for about half an hour for it.
	 *	Anyways, someone should write a tutorial on using configure() or something.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void configure()
	{
		//with no real available documentation, the best way to figure out this part is:
		//	1. Reflection
		//	2. Trial/Error
		//	3. Go to the Round Tables and whine to Pops until someone explains it
		//As such, I will probably include my Reflector to this package just for other
		//plugin writers to use.
		ConfigPanel frame = new ConfigPanel();
		frame.pack();
		frame.show();
		LocalPreferences prefs = LocalPreferences.getInstance();
		textArea.setForeground(prefs.getColor("kawigi.editor.foreground"));
		textArea.setCaretColor(textArea.getForeground());
		textArea.setBackground(prefs.getColor("kawigi.editor.background"));
		textArea.setSelectionColor(prefs.getColor("kawigi.editor.SelectionColor"));
		textArea.setSelectedTextColor(prefs.getColor("kawigi.editor.SelectedTextColor"));
		textArea.setFont(new Font(prefs.getProperty("kawigi.editor.font"), Font.PLAIN, Integer.parseInt(prefs.getProperty("kawigi.editor.font.size"))));
		localCodePane.setBackground(textArea.getBackground());
		localCodePane.setForeground(textArea.getForeground());
		localCodePane.setCaretColor(textArea.getCaretColor());
		localCodePane.setFont(textArea.getFont());
		localCodePane.setSelectionColor(textArea.getSelectionColor());
		localCodePane.setSelectedTextColor(textArea.getSelectedTextColor());
		testingPane.setForeground(prefs.getColor("kawigi.testing.foreground"));
		testingPane.setCaretColor(testingPane.getForeground());
		testingPane.setBackground(prefs.getColor("kawigi.testing.background"));
		testingPane.setFont(new Font(prefs.getProperty("kawigi.testing.font"), Font.PLAIN, Integer.parseInt(prefs.getProperty("kawigi.testing.font.size"))));
		useStdUI = prefs.isTrue("kawigi.stdui");
		if (timer == null && prefs.isTrue("kawigi.timer.show"))
		{
			timer = new ProblemTimer();
			mainPanel.add(timer, BorderLayout.WEST);
			mainPanel.revalidate();
		}
		else if (timer != null && !prefs.isTrue("kawigi.timer.show"))
		{
			timer.stop();
			mainPanel.remove(timer);
			timer = null;
			mainPanel.revalidate();
		}
		tempdir = new File(prefs.getProperty("kawigi.localpath"));
		if (!tempdir.exists())
			tempdir.mkdir();
	}
	
	/**
	 *	Verifies or sets several properties used by parts of the editor to set up.
	 *	
	 *	For awhile, I was using this, but the constructor pretty much checks to make 
	 *	I have all my configurations intact.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void install()
	{
	}
	
	/**
	 *	Stores away edits so they can be undone later.
	 **/
	public void undoableEditHappened(UndoableEditEvent e)
	{
		//Remember the edit and update the menus
		undo.addEdit(e.getEdit());
	}
	
	/**
	 *	Creates, saves, and returns a popup menu for the given JTextPane to display.
	 *	
	 *	This has the snippet menu, search/replace stuff, and (if text is selected)
	 *	the add-snippet thingy.  I know the UI is non-standard, maybe I'll provide
	 *	a way to configure it to use dialogs instead of funny panels on a popup menu.
	 **/
	private JPopupMenu getPopup(NoWrapJTextPane pane)
	{
		currentPopupThing = pane;
		popup = new JPopupMenu();
		popup.add(baseCategory);
		if (pane.getSelectedText() != null)
		{
			JMenu weirdMenu = new JMenu("Add Snippet");
			weirdMenu.add(snippetPanel);
			popup.add(weirdMenu);
			String code = pane.getSelectedText();
			codeSelection = code;
			if (code.indexOf('(') >= 0)
			{
				String name = code.substring(0, code.indexOf('(')).trim();
				if (name.indexOf('\n') >= 0)
					name = name.substring(name.lastIndexOf('\n')+1).trim();
				if (name.indexOf(' ') >= 0)
					name = name.substring(name.lastIndexOf(' ')+1).trim();
				nameField.setText(name);
			}
			else
				nameField.setText("");
		}
		JMenu replaceMenu = new JMenu("Search/Replace");
		replaceMenu.add(replacePanel);
		popup.add(replaceMenu);
		JMenu varReplaceMenu = new JMenu("Change var name");
		varReplaceMenu.add(varReplacePanel);
		popup.add(varReplaceMenu);
		return popup;
	}
	
	/**
	 *	Listens to mouse events on the text pane, so it knows when to create the
	 *	popup menu.
	 **/
	public void mousePressed(MouseEvent e)
	{
		if (e.isPopupTrigger())
		{
			JPopupMenu popup = getPopup((NoWrapJTextPane)e.getComponent());
			popup.show(e.getComponent(), e.getX(), e.getY());
		}
	}
	
	/**
	 *	Listens to mouse events on the text pane, so it knows when to create the
	 *	popup menu.
	 **/
	public void mouseReleased(MouseEvent e)
	{
		if (e.isPopupTrigger())
		{
			JPopupMenu popup = getPopup((NoWrapJTextPane)e.getComponent());
			popup.show(e.getComponent(), e.getX(), e.getY());
		}
	}
	
	/**
	 *	Required by the MouseListener interface.
	 **/
	public void mouseClicked(MouseEvent e){}
	
	/**
	 *	Required by the MouseListener interface.
	 **/
	public void mouseEntered(MouseEvent e){}
	
	/**
	 *	Required by the MouseListener interface.
	 **/
	public void mouseExited(MouseEvent e){}
	
	
    /**
     *	Invoked when a component gains the keyboard focus.
     *	
     *	Herein lies a hack to avoid moving the focus from the text editor to the tabs at the top
     *	when the user pushes ctrl+up.  Mainly because ctrl+up is a quick-view scroll up.
     **/
    public void focusGained(FocusEvent e)
    {
    	textArea.requestFocus();
    }

    /**
     *	Invoked when a component loses the keyboard focus.
     **/
    public void focusLost(FocusEvent e)
    {
    }
    
	/**
	 *	Notifies me of insertions into the text editor.
	 **/
    public void insertUpdate(DocumentEvent e)
    {
    }

	/**
	 *	Notifies me of removals from the text editor.
	 **/
    public void removeUpdate(DocumentEvent e)
    {
    }

	/**
	 *	Notifies me of non-insert/remove sorts of edits in the editor.
	 **/
    public void changedUpdate(DocumentEvent e)
    {
    }
	
	/**
	 *	Run the plugin as a standalone application.
	 *	
	 *	This is mostly for view debugging, since it won't do anything exciting, and
	 *	many of the features won't completely work (because the plugin methods won't
	 *	be called).
	 **/
	public static void main(String[] args)
	{
		JFrame frame = new JFrame("KawigiEdit " + VERSION);
		frame.getContentPane().add(new KawigiEdit().getEditorPanel());
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setSize(700, 500);
		frame.show();
	}
}
