package kawigi.util;
import java.io.*;
import javax.swing.text.*;

/**
 *	I couldn't do local compiling and testing without this.
 *	
 *	This contains and manages a process and forwards its output using ProcessOutput
 *	objects.
 **/
public class ProcessContainer implements ConsoleDisplay
{
	private Process p;
	private JTextComponent outputComponent;
	private ProcessOutput stdout, stderr;
	private int exitVal;
	private boolean done;
	
	/**
	 *	Creates a new ProcessContainer for p that forwards its standard output/error
	 *	streams into <code>output</code>.
	 *	
	 *	Note that it will clear output if it isn't already clear.
	 *	
	 *	This process will be forcibly killed according to the current user settings.
	 **/
	public ProcessContainer(Process p, JTextComponent output)
	{
		this.p = p;
		outputComponent = output;
		output.setText("");
		new KillThread(this).start();
	}
	
	/**
	 *	Creates a new ProcessContainer for p that forwards its standard output/error
	 *	streams into <code>output</code>.
	 *	
	 *	Note that it will clear output if it isn't already clear.
	 *	
	 *	If doTimeout is true, the process will be killed according to the current user
	 *	settings for the time limit.
	 **/
	public ProcessContainer(Process p, JTextComponent output, boolean doTimeout)
	{
		this.p = p;
		outputComponent = output;
		output.setText("");
		if (doTimeout)
			new KillThread(this).start();
	}
	
	/**
	 *	Starts the threads that listen to the output and error streams of the process.
	 **/
	public synchronized void start()
	{
		stdout = new ProcessOutput(p.getInputStream(), this);
		stderr = new ProcessOutput(p.getErrorStream(), this);
		stdout.start();
		stderr.start();
	}
	
	/**
	 *	Finishes and stores the exit value for the process.
	 **/
	public synchronized void waitFor()
	{
		try
		{
			exitVal = p.waitFor();
			done = true;
		}
		catch (InterruptedException ex)
		{
			println("***\nInterrupted***\n");
		}
	}
	
	/**
	 *	Returns true if this process has finished.
	 **/
	public synchronized boolean isDone()
	{
		return done || (stderr != null && stdout != null && stderr.isDone() && stdout.isDone());
	}
	
	/**
	 *	Forceably kills the process.
	 *	
	 *	You'll be glad sometimes that this option exists.
	 **/
	public void kill()
	{
		p.destroy();
	}
	
	/**
	 *	Returns the exit value of the process, waiting for it to finish if it hasn't yet.
	 **/
	public int endVal()
	{
		if (!done)
			waitFor();
		if (done)
			return exitVal;
		return -1;
	}
	
	/**
	 *	Outputs a line of text to the process's output component.
	 **/
	public void println(String s)
	{
		outputComponent.setText(outputComponent.getText() + s + "\n");
	}
	
	/**
	 *	Outputs a string of text to the process's output component.
	 **/
	public void print(String s)
	{
		outputComponent.setText(outputComponent.getText() + s);
	}
}
