package kawigi.editor;
import javax.swing.text.*;
import java.awt.*;
import java.util.*;
import com.topcoder.client.contestApplet.common.LocalPreferences;

/**
 *	Implementation of a Visual Basic implementation of a View that hilights syntax attributes.
 **/
public class VBView extends GenericView
{
	protected static Hashtable colorHash;
	protected static Color keywordColor, typeColor, operatorColor, classColor, stringColor, commentColor, directiveColor;
	
	static
	{
		colorHash = new Hashtable();
		initColors();
	}
	
	/**
	 *	Somone who knows VB better than me should hack this up sometime, I probably have
	 *	it all messed up.
	 *	
	 *	Particularly in the "class" section, I added several standard function names
	 *	that I don't think were keywords, but are still "language features", and that
	 *	might have been overboard.
	 **/
	public static void initColors()
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		if (prefs.getProperty("kawigi.editor.KeywordColor") == null)
			prefs.setColor("kawigi.editor.KeywordColor", keywordColor = new Color(191, 191, 0));
		else
			keywordColor = prefs.getColor("kawigi.editor.KeywordColor");
		if (prefs.getProperty("kawigi.editor.TypeColor") == null)
			prefs.setColor("kawigi.editor.TypeColor", typeColor = new Color(127, 127, 255));
		else
			typeColor = prefs.getColor("kawigi.editor.TypeColor");
		if (prefs.getProperty("kawigi.editor.OperatorColor") == null)
			prefs.setColor("kawigi.editor.OperatorColor", operatorColor = new Color(191, 63, 63));
		else
			operatorColor = prefs.getColor("kawigi.editor.OperatorColor");
		if (prefs.getProperty("kawigi.editor.ClassColor") == null)
			prefs.setColor("kawigi.editor.ClassColor", classColor = new Color(191, 63, 191));
		else
			classColor = prefs.getColor("kawigi.editor.ClassColor");
		if (prefs.getProperty("kawigi.editor.StringColor") == null)
			prefs.setColor("kawigi.editor.StringColor", stringColor = new Color(255, 0, 0));
		else
			stringColor = prefs.getColor("kawigi.editor.StringColor");
		if (prefs.getProperty("kawigi.editor.CommentColor") == null)
			prefs.setColor("kawigi.editor.CommentColor", commentColor = new Color(127, 255, 127));
		else
			commentColor = prefs.getColor("kawigi.editor.CommentColor");
		if (prefs.getProperty("kawigi.editor.DirectiveColor") == null)
			prefs.setColor("kawigi.editor.DirectiveColor", directiveColor = new Color(255, 127, 127));
		else
			directiveColor = prefs.getColor("kawigi.editor.DirectiveColor");
		
		String[] keywords = new String[]{"AddHandler", "Alias", "Ansi", "As", "Assembly", "Auto", "ByRef",
			"ByVal", "Call", "Case", "Catch", "Class", "Clear", "Compare", "Const", "Declare", "Default", "Delegate", "Dim", "DirectCast", "Do", "Each", "Else", "ElseIf",
			"End", "Enum", "Erase", "Error", "Event", "Exit", "Explicit", "Finally", "For", "Friend", "Function", "Get", "GoSub", "GoTo", "Handles",
			"If", "Implements", "Imports", "In", "Inherits", "Interface", "Let", "Lib", "Loop", "Module",
			"MustInherit", "MustOverride", "MyBase", "MyClass", "Namespace", "New", "Next", "NotInheritable", "NotOverridable",
			"Off", "On", "Option", "Optional", "Overloads", "Overridable", "Overrides", "ParamArray", "Preserve", "Private", "Property",
			"Protected", "Public", "Raise", "RaiseEvent", "ReadOnly", "ReDim", "REM", "RemoveHandler", "Resume", "Return", "Select", "Set", "Shadows",
			"Shared", "Static", "Step", "Stop", "Strict", "Structure", "Sub", "SyncLock", "Then", "Throw", "To", "Try", "TypeOf",
			"Unicode", "Until", "Variant", "When", "While", "With", "WithEvents", "WriteOnly"};
		String[] types = new String[]{"Boolean", "Byte", "Char", "Date", "Decimal", "Double", "False", "Integer", "Long", "Object", "Short", "Single", "String", "True", "Me", "Nothing"};
		String[] operators = new String[]{"AddressOf", "And", "AndAlso", "GetType", "Is", "Like", "Mod", "Not", "Or", "OrElse", "Xor", "^", "=", "\\", "/", "&", "*", "+", "-", "<", ">"};
		//classes, interfaces, structs, delegates and enumerations in System, System.Collections, System.Collections.Specialized,
		//System.Text and System.Text.RegularExpressions, as well as several object-y function-y structures in Visual Basic.  I'm
		//likely only mostly right on what should be in here:
		String[] classes = new String[]{"AppWinStyle", "CallType", "CompareMethod", "DateFormat", "DateInterval", "DueDate", "FileAttribute",
			"FirstDayOfWeek", "FirstWeekOfYear", "MsgBoxResult", "MsgBoxStyle", "OpenAccess", "OpenMode", "OpenShare", "ControlChars",
			"TriState", "VariantType", "VBStrConv", "COMClassAttribute", "VBFixedArrayAttribute", "VBFixedStringAttribute", "Collection",
			"Choose", "CBool", "CByte", "CChar", "CDate", "CDec", "CDbl", "CInt", "CLng", "CObj", "CShort", "CSng", "CStr", "CType",
			"Chr", "ChrW", "Format", "LCase", "UCase", "DateSerial", "DateValue", "Hex", "Oct", "Str", "Fix", "Int", "Day", "Month", "Weekday",
			"Year", "Hour", "Minute", "Second", "Asc", "AscW", "Val", "TimeSerial", "TimeValue", "IsArray", "IsDate", "IsDbNull", "IsError",
			"IsNothing", "IsNumeric", "IsReference", "Now", "Today", "TimeOfDay", "DateAdd", "DateDiff", "DatePart", "DateSerial", "DateValue",
			"MonthName", "WeekDayName", "DateString", "TimeString", "Timer", "ChDir", "ChDrive", "FileCopy", "MkDir", "RmDir", "Rename",
			"CurDir", "FileDateTime", "GetAttr", "FileLen", "Dir", "SetAttr", "GetException", "Err", "Erl", "LastDLLError", "DDB", "SLN",
			"SYD", "FV", "Rate", "IRR", "MIRR", "NPer", "IPmt", "Pmt", "PPmt", "NPV", "PV", "FileOpen", "FileClose", "Reset", "Print", "Spc",
			"Tab", "FileWidth", "FileCopy", "EOF", "FileAttr", "FreeFile", "Loc", "LOF", "Seek", "Kill", "Lock", "Unlock", "FileGet",
			"FileGetObject", "Input", "InputString", "LineInput", "FilePut", "FilePutObject", "Write", "Atn", "Cos", "Sin", "Tan", "Exp",
			"Log", "Sqr", "Randomize", "Rnd", "Abs", "Sgn", "AppActivate", "Shell", "CallByName", "Beep", "Command", "CreateObject", "GetObject",
			"QBColor", "RGB", "InputBox", "MsgBox", "DeleteSetting", "GetSetting", "GetAllSettings", "SaveSetting", "StrComp", "StrConv",
			"InStrRev", "StrReverse", "Space", "StrDup", "Len", "FormatCurrency", "FormatDateTime", "FormatNumber", "FormatPercent",
			"InStr", "Left", "LTrim", "Mid", "Right", "RTrim", "Trim", "Replace", "Filter", "Split", "Join", "Activator", "AppDomain", "AppDomainSetup",
			"AppDomainUnloadedException", "ApplicationException", "ArgumentException", "ArgumentNullException", "ArgumentOutOfRangeException",
			"ArithmeticException", "Array", "ArrayTypeMismatchException", "AssemblyLoadEventArgs", "Attribute", "AttributeUsageAttribute",
			"BadImageFormatException", "BitConverter", "Buffer", "CannotUnloadAppDomainException", "CharEnumerator", "CLSCompliantAttribute",
			"Console", "ContextBoundObject", "ContextMarshalException", "ContextStaticAttribute", "Convert", "DBNull", "Delegate", "DivideByZeroException",
			"DllNotFoundException", "DuplicateWaitObjectException", "EntryPointNotFoundException", "Enum", "Environment", "EventArgs",
			"Exception", "ExecutionEngineException", "FieldAccessException", "FlagsAttribute", "FormatException", "GC", "IndexOutOfRangeException",
			"InvalidCastException", "InvalidOperationException", "InvalidProgramException", "LoaderOptimizationAttribute", "LocalDataStoreSlot",
			"MarshalByRefObject", "Math", "MemberAccessException", "MethodAccessException", "MissingFieldException", "MissingMemberException",
			"MissingMethodException", "MTAThreadAttribute", "MulticastDelegate", "MulticastNotSupportedException", "NonSerializedAttribute",
			"NotFiniteNumberException", "NotImplementedException", "NotSupportedException", "NullReferenceException", "Object", "ObjectDisposedExeption",
			"ObsoleteException", "OperatingSystem", "OutOfMemoryException", "OverflowException", "ParamArrayAttribute", "PlatformNotSupportedException",
			"Random", "RankException", "ResolveEventArgs", "SerializableAttribute", "StackOverflowException", "STAThreadAttribute", "String",
			"SystemException", "ThreadStaticAttribute", "TimeZone", "Type", "TypeInitializationException", "TypeLoadException", "TypeUnloadedException",
			"UnauthorizedAccessException", "UnhandledExceptionEventArgs", "Uri", "UriBuilder", "UriFormatException", "ValueType", "Version",
			"WeakReference", "IAppDomainSetup", "IAsyncResult", "ICloneable", "IComparable", "IConvertible", "ICustomerFormatter", "IDisposable",
			"IFormatProvider", "IFormattable", "IServiceProvider", "_AppDomain", "ArgIterator", "Boolean", "Byte", "Char", "DateTime", "Decimal",
			"Double", "Guid", "Int16", "Int32", "Int64", "IntPtr", "RuntimeArgumentHandle", "RuntimeFieldHandle", "RuntimeMethodHandle",
			"RuntimeTypeHandle", "SByte", "Single", "TimeSpan", "TypedReference", "UInt16", "UInt32", "UInt64", "UIntPtr", "Void", "AssemblyLoadEventHandler",
			"AsyncCallback", "CrossAppDomainDelegate", "EventHandler", "ResolveEventHandler", "UnhandledExceptionEventHandler", "AttributeTargets",
			"DayOfWeek", "LoaderOptimization", "PlatformID", "TypeCode", "UriHostNameType", "UriPartial", "ArrayList", "BitArray", "CaseInsensitiveComparer",
			"CaseInsensitiveHashCodeProvider", "CollectionBase", "Comparer", "DictionaryBase", "Hashtable", "Queue", "ReadOnlyCollectionBase",
			"SortedList", "Stack", "ICollection", "IComparer", "IDictionary", "IDictionaryEnumerator", "IEnumerable", "IEnumerator", "IHashCodeProvider",
			"IList", "DictionaryEntry", "CollectionsUtil", "HybridDictionary", "ListDictionary", "NameObjectCollectionBase", "NameValueCollection",
			"StringCollection", "StringDictionary", "StringEnumerator", "BitVector32", "ASCIIEncoding", "Decoder", "Encoder", "Encoding", "StringBuilder",
			"UnicodeEncoding", "UTF7Encoding", "UTF8Encoding", "Capture", "CaptureCollection", "Group", "GroupCollection", "Match", "MatchCollection",
			"Regex", "RegexCompilationInfo", "MatchEvaluator", "RegexOptions"};
		String[] directives = new String[]{"#Const", "#If", "#Then", "#Region", "#ExternalSource", "#End"};
		
		//note that I put everything in lower case here to support case-insensitivity.
		for (int i=0; i<keywords.length; i++)
			colorHash.put(keywords[i].toLowerCase(), keywordColor);
		for (int i=0; i<types.length; i++)
			colorHash.put(types[i].toLowerCase(), typeColor);
		for (int i=0; i<operators.length; i++)
			colorHash.put(operators[i].toLowerCase(), operatorColor);
		for (int i=0; i<classes.length; i++)
			colorHash.put(classes[i].toLowerCase(), classColor);
		for (int i=0; i<directives.length; i++)
			colorHash.put(directives[i].toLowerCase(), directiveColor);
	}
	
	/**
	 *	Creates an instance of VBView and passes along the Element.
	 **/
	public VBView(Element e)
	{
		super(e);
	}
	
	/**
	 *	Overridden from PlainView - this method gets called to render every element
	 *	of unselected text.
	 **/
	protected int drawUnselectedText(Graphics g, int x, int y, int p0, int p1) throws BadLocationException
	{
		String s = getDocument().getText(p0, p1 - p0);
		String before = getDocument().getText(0, p0);
		boolean inComment = false;
		for (int i=0; i<before.length(); i++)
		{
			if (inComment && before.charAt(i) == '/' && i > 0 && before.charAt(i-1) == '*')
				inComment = false;
			else if (!inComment && before.charAt(i) == '*' && i > 0 && before.charAt(i-1) == '/')
				inComment = true;
			else if (!inComment && before.charAt(i) == '\"')
				for (i = i+1; i<before.length() && before.charAt(i) != '\"' && before.charAt(i) != '\r' && before.charAt(i) != '\n'; i++)
					;
			else if (!inComment && before.charAt(i) == '\'')
				for (i = i+1; i<before.length() && before.charAt(i) != '\'' && before.charAt(i) != '\r' && before.charAt(i) != '\n'; i++)
					;
			else if (!inComment && before.charAt(i) == '/' && i > 0 && before.charAt(i-1) == '/')
				for (i = i+1; i<before.length() && before.charAt(i) != '\r' && before.charAt(i) != '\n'; i++)
					;
		}
		return drawTabbedText(s, x, y, g, p0, false);
	}
	
	/**
	 *	Overridden from PlainView - this method gets called to render every element
	 *	of selected text, so that I could change how it's displayed if I want.
	 **/
	protected int drawSelectedText(Graphics g, int x, int y, int p0, int p1) throws BadLocationException
	{
		String s = getDocument().getText(p0, p1 - p0);
		String before = getDocument().getText(0, p0);
		return drawTabbedText(s, x, y, g, p0, true);
	}
	
	/**
	 *	Renders the text segment onto the given Grapnics context.
	 *	
	 *	Originally, this looked a lot like javax.swing.text.Utilities.drawTabbedText
	 *	with a little bit of tokenizing, but now it's pretty much modified to my own
	 *	style.  Unfortunately, at some point in time, my indentation got all weird
	 *	(probably as a result of me using tabs and the original implementation mixing
	 *	spaces and tabs), so there may be some weird-looking parts left in the code.
	 *	
	 *	This is the magic of my syntax hilighting.
	 **/
	protected int drawTabbedText(String s, int x, int y, Graphics g, int startOffset, boolean selected)
	{
		if (selected)
			g.setColor(((JTextComponent)getContainer()).getSelectedTextColor());
		FontMetrics metrics = g.getFontMetrics();
		int flushIndex = 0;
		for (int i = 0; i < s.length(); i++)
		{
			//parse a string:
			if (s.charAt(i) == '\"')
			{
	    		if (flushIndex < i)
	    		{
					if (!selected)
			    		g.setColor(getColor(s.substring(flushIndex, i)));
		    		g.drawString(s.substring(flushIndex, i), x, y);
		    		x += metrics.stringWidth(s.substring(flushIndex, i));
		   	 		flushIndex = i;
		    	}
				i++;
				for (; i<s.length() && s.charAt(i) != '\"' && s.charAt(i) != '\n' && s.charAt(i) != '\r'; i++)
				{
					if (s.charAt(i) == '\t')
					{
						if (i > flushIndex)
						{
							if (!selected)
		    						g.setColor(stringColor);
							g.drawString(s.substring(flushIndex, i), x, y);
		    				x += metrics.stringWidth(s.substring(flushIndex, i));
						}
						flushIndex = i + 1;
						x = (int)nextTabStop((float)x, startOffset+i);
					}
					//There's apparently only one escape sequence in VB.  Weird, huh?
					if (s.charAt(i) == '\"' && i+1 < s.length() && s.charAt(i) == '\"')
						i++;
				}
				if (!selected)
			    	g.setColor(stringColor);
				g.drawString(s.substring(flushIndex, i), x, y);
		    	x += metrics.stringWidth(s.substring(flushIndex, i));
				if (i < s.length() && s.charAt(i) == '\"')
				{
					g.drawString(s.substring(i, i+1), x, y);
			    	x += metrics.charWidth(s.charAt(i));
			    }
	    		flushIndex = i+1;
			}
			else if (s.charAt(i) == '\'')	//parsing VB comments.
			{
	    		if (flushIndex < i)
	    		{
					if (!selected)
						g.setColor(getColor(s.substring(flushIndex, i)));
		   	 		g.drawString(s.substring(flushIndex, i), x, y);
		    		x += metrics.stringWidth(s.substring(flushIndex, i));
					flushIndex = i;
				}
				i++;
				for (; i<s.length() && s.charAt(i) != '\n' && s.charAt(i) != '\r'; i++)
				{
					if (s.charAt(i) == '\t')
					{
						if (i > flushIndex)
						{
							if (!selected)
		    					g.setColor(commentColor);
							g.drawString(s.substring(flushIndex, i), x, y);
		    				x += metrics.stringWidth(s.substring(flushIndex, i));
						}
						flushIndex = i + 1;
						x = (int)nextTabStop((float)x, startOffset+i);
					}
				}
				if (!selected)
					g.setColor(commentColor);
				//some people may let this bug them, but I like having a tribute
				//to myself in my plugin, so those people can hack it out or put
				//up with it.
				if (s.substring(flushIndex, i).endsWith("[KawigiEdit]"))
			   	{
			   		if (!s.substring(flushIndex, i).equals("[KawigiEdit]"))
			   		{
			   			g.drawString(s.substring(flushIndex, s.indexOf("[KawigiEdit]", flushIndex)), x, y);
			   			x += metrics.stringWidth(s.substring(flushIndex, s.indexOf("[KawigiEdit]", flushIndex)));
			   		}
			   		Font oldFont = g.getFont();
			   		g.setFont(oldFont.deriveFont(Font.BOLD));
			    	if (!selected)
				   		g.setColor(getContainer().getForeground());
			    	g.drawString("[KawigiEdit]", x, y);
			    	if (!selected)
				   		g.setColor(Color.red);
		    		g.drawString("Edit", x+g.getFontMetrics().stringWidth("[Kawigi"), y);
		    		x += g.getFontMetrics().stringWidth("[KawigiEdit]");
			   		g.setFont(oldFont);
			   	}
			   	else
			   	{
					g.drawString(s.substring(flushIndex, i), x, y);
		    		x += metrics.stringWidth(s.substring(flushIndex, i));
		    	}
				if (i < s.length() && s.charAt(i) == '\'')
				{
					g.drawString(s.substring(i, i+1), x, y);
			   		x += metrics.charWidth(s.charAt(i));
			   	}
		    	flushIndex = i+1;
			}
			//Tabs are basically non-displayable characters, so I need to flush and
			//move where I'm drawing.
			else if (s.charAt(i) == '\t')
			{
				if (i > flushIndex)
				{
					if (!selected)
						g.setColor(getColor(s.substring(flushIndex, i)));
					g.drawString(s.substring(flushIndex, i), x, y);
		    		x += metrics.stringWidth(s.substring(flushIndex, i));
				}
				flushIndex = i + 1;
				x = (int) nextTabStop((float) x, startOffset+i);
			}
			//end of line...
			else if ((s.charAt(i) == '\n') || (s.charAt(i) == '\r'))
			{
				if (i > flushIndex)
				{
					if (!selected)
			    		g.setColor(getColor(s.substring(flushIndex, i)));
					g.drawString(s.substring(flushIndex, i), x, y);
		    		x += metrics.stringWidth(s.substring(flushIndex, i));
				}
				flushIndex = i + 1;
	    	}
	    	//parsing compiler directives:
	    	else if (s.charAt(i) == '#')
	    	{
	    		if (flushIndex < i)
	    		{
					if (!selected)
			    		g.setColor(getColor(s.substring(flushIndex, i)));
		    		g.drawString(s.substring(flushIndex, i), x, y);
		    		x += metrics.stringWidth(s.substring(flushIndex, i));
		   	 		flushIndex = i;
		    	}
				i++;
				for (; i<s.length() && s.charAt(i) != '\n' && s.charAt(i) != '\r' && (s.charAt(i) != '/' || i+1 >= s.length() || (s.charAt(i+1) != '/' && s.charAt(i+1) != '*')); i++)
				{
					if (s.charAt(i) == '\t')
					{
						if (i > flushIndex)
						{
							if (!selected)
		    					g.setColor(directiveColor);
							g.drawString(s.substring(flushIndex, i), x, y);
		    				x += metrics.stringWidth(s.substring(flushIndex, i));
						}
						flushIndex = i + 1;
						x = (int)nextTabStop((float)x, startOffset+i);
					}
				}
				i--;
				if (i >= flushIndex)
				{
					if (!selected)
				    	g.setColor(getColor(s.substring(flushIndex, i+1).split("[ \\t\\n\\r\\f]")[0]));
					g.drawString(s.substring(flushIndex, i+1), x, y);
		    		x += metrics.stringWidth(s.substring(flushIndex, i+1));
				}
				flushIndex = i+1;
	    	}
	    	//all this stuff is part of a normal "token"
			else if (Character.isLetterOrDigit(s.charAt(i)) || s.charAt(i) == '_')
			{
			}
		    else	//otherwise flush!
		    {
		    	//if I have stuff in the buffer, display it first...
		    	if (flushIndex < i)
		    	{
					if (!selected)
			    		g.setColor(getColor(s.substring(flushIndex, i)));
		    		g.drawString(s.substring(flushIndex, i), x, y);
		    		x += metrics.stringWidth(s.substring(flushIndex, i));
		    	}
				if (!selected)
		    		g.setColor(getColor(s.substring(i, i+1)));
		    	//then do the current character!
				g.drawString(s.substring(i, i+1), x, y);
				flushIndex = i+1;
				x += metrics.charWidth(s.charAt(i));
			}
		}
		//flush if I haven't yet.
		if (flushIndex < s.length())
		{
			if (!selected)
				g.setColor(getColor(s.substring(flushIndex, s.length())));
			g.drawString(s.substring(flushIndex, s.length()), x, y);
		    x += metrics.stringWidth(s.substring(flushIndex, s.length()));
		}
		return x;
	}
	
	/**
	 *	Returns the color to use on the given token.
	 *	
	 *	This will return the default foreground color, if no special color is 
	 *	assigned to that token.
	 **/
	protected Color getColor(String word)
	{
		if (colorHash.containsKey(word.toLowerCase()))
			return (Color)colorHash.get(word.toLowerCase());
		else
			return getContainer().getForeground();
	}
	
	/**
	 *	Returns an <code>ArrayList</code> of <code>Interval</code> objects representing code block intervals
	 *	in the document.
	 *	
	 *	This version is specifically for finding blocks in the Visual Basic language.
	 **/
	public ArrayList getIntervals()
	{
		ArrayList ret = new ArrayList();
		try
		{
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"class", "end class"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"module", "end module"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"sub", "end sub"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"for", "next"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"if", "end if"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"with", "end with"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"structure", "end structure"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"function", "end function"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"synclock", "end synclock"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"try", "end try"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"get", "end get"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"set", "end set"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"property", "end property"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"select", "end select"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"do", "loop"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"while", "end while"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"(", ")"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"#region", "#end region"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"#externalsource", "#end externalsource"});
			parseIndex = 0;
			lineIndex = 1;
			findIntervals(ret, getDocument().getText(0, getDocument().getLength()).toLowerCase(), new String[]{"#if", "#end if"});
		}
		catch (BadLocationException ex)
		{
		}
		return ret;
	}
	
	/**
	 *	Helper function to help with <code>getIntervals</code>.
	 *	
	 *	Puts the intervals into the list in an in-order traversal of blocks between startEnd[0] and startEnd[1].
	 *	This is modified from the C-syntax-based one in GenericView to work better with VB's string and comment
	 *	syntax.
	 **/
	protected void findIntervals(ArrayList list, String text, String[] startEnd)
	{
		int startline = lineIndex;
		int startIndex = parseIndex;
		parseIndex++;
		while (parseIndex < text.length())
		{
			if (text.charAt(parseIndex) == '\n')
				lineIndex ++;
			else if (text.substring(parseIndex).startsWith(startEnd[0]))
			{
				findIntervals(list, text, startEnd);
			}
			else if (text.substring(parseIndex).startsWith(startEnd[1]))
			{
				Interval in = findName(text, startIndex, startline, lineIndex, parseIndex, startEnd);
				if (in != null)
					list.add(in);
				return;
			}
			else if (text.charAt(parseIndex) == '\"')
			{
				parseIndex++;
				while (parseIndex < text.length() && text.charAt(parseIndex) != '\"' && text.charAt(parseIndex) != '\n' && text.charAt(parseIndex) != '\r')
					parseIndex++;
			}
			else if (text.charAt(parseIndex) == '\'')
			{
				while (parseIndex < text.length() && text.charAt(parseIndex) != '\n' && text.charAt(parseIndex) != '\r')
					parseIndex++;
				parseIndex--;
			}
			parseIndex++;
		}
	}
}
