package kawigi;
import kawigi.config.*;
import kawigi.challenge.*;
import kawigi.snippet.*;
import kawigi.util.*;
import kawigi.editor.*;
import java.lang.reflect.*;
import java.io.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.undo.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import com.topcoder.shared.language.*;
import com.topcoder.shared.problem.*;
import com.topcoder.client.contestant.*;
import com.topcoder.client.contestApplet.common.LocalPreferences;
import kawigi.language.*;

/**
 *	KawigiEdit - an open-source, fully-featured editor plugin for the Top Coder Arena.  It was
 *		written by me for me, so don't be surprised if the features aren't in line with what you
 *		want.  If you want something that's not here, talk to me, and either I will add it, or
 *		if I don't care, you can.
 **/
public class KawigiEdit implements ActionListener, CaretListener, KeyListener, MouseListener, UndoableEditListener
{
	protected NoWrapJTextPane textArea, localCodePane, currentPopupThing;
	protected JTextArea testingPane;
	protected JTabbedPane tabs;
	protected JPanel mainPanel, snippetPanel, replacePanel, varReplacePanel;
	protected JFileChooser localFileChooser;
	private ProblemComponent component;
	private JButton generateButton, testButton, saveButton, killButton, snippetButton, replaceButton, varReplaceButton, challengeToolsButton;
	private JLabel lineLabel;
	private File tempdir;
	private ProcessContainer proc;
	private JMenu fileMenu;
	private JMenuItem openFileDialogItem;
	private JPopupMenu popup;
	private Category baseCategory;
	private JTextField nameField, categoryField, replaceField, replaceWithField, varReplaceField, varReplaceWithField;
	private String codeSelection;
	private UndoManager undo;
	private ExtendedLanguage language;
	private boolean useStdUI;
	
	/**
	 *	Initializes KawigiEdit and checks to make sure certain settings are available.
	 **/
	public KawigiEdit()
	{
		LocalPreferences prefs = LocalPreferences.getInstance();
		if (prefs.getProperty("kawigi.stdui") != null)
			useStdUI = prefs.isTrue("kawigi.stdui");
		else
			prefs.setTrue("kawigi.stdui", useStdUI = false);
		if (prefs.getProperty("kawigi.localpath") != null)
			tempdir = new File(prefs.getProperty("kawigi.localpath"));
		else
		{
			tempdir = new File("testprograms");
			prefs.setProperty("kawigi.localpath", tempdir.getPath());
		}
		if (!tempdir.exists())
			tempdir.mkdir();
		textArea = new NoWrapJTextPane();
		if (prefs.getProperty("kawigi.editor.background") != null)
			textArea.setBackground(prefs.getColor("kawigi.editor.background"));
		else
		{
			textArea.setBackground(Color.black);
			prefs.setColor("kawigi.editor.background", Color.black);
		}
		if (prefs.getProperty("kawigi.editor.foreground") != null)
		{
			textArea.setForeground(prefs.getColor("kawigi.editor.foreground"));
			textArea.setCaretColor(prefs.getColor("kawigi.editor.foreground"));
		}
		else
		{
			textArea.setForeground(Color.white);
			textArea.setCaretColor(Color.white);
			prefs.setColor("kawigi.editor.foreground", Color.white);
		}
		String fontface;
		if (prefs.getProperty("kawigi.editor.font") != null)
			fontface = prefs.getProperty("kawigi.editor.font");
		else
			prefs.setProperty("kawigi.editor.font", fontface = "Monospaced");
		int fontsize;
		if (prefs.getProperty("kawigi.editor.font.size") != null)
			fontsize = prefs.getFontSize("kawigi.editor.font.size");
		else
			prefs.setFontSize("kawigi.editor.font.size", Integer.toString(fontsize = 12));
		textArea.setFont(new Font(fontface, Font.PLAIN, 12));
		textArea.addCaretListener(this);
		textArea.addKeyListener(this);
		textArea.addMouseListener(this);
		textArea.getStyledDocument().addUndoableEditListener(this);
		textArea.setDragEnabled(true);
		LineNumbers lines = new LineNumbers(textArea);
		JPanel numberedTextPanel = new JPanel(new BorderLayout());
		numberedTextPanel.add(lines, BorderLayout.WEST);
		numberedTextPanel.add(textArea);
		tabs = new JTabbedPane();
		tabs.add(new JScrollPane(numberedTextPanel, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED), "Source Code");
		
		if (prefs.getProperty("kawigi.testing.font") != null)
			fontface = prefs.getProperty("kawigi.testing.font");
		else
			prefs.setProperty("kawigi.testing.font", fontface = "Monospaced");
		if (prefs.getProperty("kawigi.editor.font.size") != null)
			fontsize = prefs.getFontSize("kawigi.testing.font.size");
		else
			prefs.setFontSize("kawigi.testing.font.size", Integer.toString(fontsize = 12));
		testingPane = new JTextArea();
		testingPane.setText("No results yet");
		if (prefs.getProperty("kawigi.testing.background") != null)
			testingPane.setBackground(prefs.getColor("kawigi.testing.background"));
		else
		{
			testingPane.setBackground(Color.white);
			prefs.setColor("kawigi.testing.background", Color.white);
		}
		if (prefs.getProperty("kawigi.testing.foreground") != null)
		{
			testingPane.setForeground(prefs.getColor("kawigi.testing.foreground"));
			testingPane.setCaretColor(prefs.getColor("kawigi.testing.foreground"));
		}
		else
		{
			testingPane.setForeground(Color.black);
			testingPane.setCaretColor(Color.black);
			prefs.setColor("kawigi.testing.foreground", Color.black);
		}
		testingPane.setEditable(false);
		testingPane.setFont(new Font(fontface, Font.PLAIN, fontsize));
		tabs.add(new JScrollPane(testingPane, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED), "Test Results");
		
		localFileChooser = new JFileChooser(tempdir);
		if (!useStdUI)
			localFileChooser.setControlButtonsAreShown(false);
		localFileChooser.addActionListener(this);
		
		localCodePane = new NoWrapJTextPane();
		localCodePane.setBackground(textArea.getBackground());
		localCodePane.setForeground(textArea.getForeground());
		localCodePane.setCaretColor(textArea.getCaretColor());
		localCodePane.setFont(textArea.getFont());
		localCodePane.setEditable(false);
		localCodePane.addMouseListener(this);
		lines = new LineNumbers(localCodePane);
		numberedTextPanel = new JPanel(new BorderLayout());
		numberedTextPanel.add(lines, BorderLayout.WEST);
		numberedTextPanel.add(localCodePane);
		JMenuBar menubar = new JMenuBar();
		fileMenu = new JMenu("File");
		if (!useStdUI)
			fileMenu.add(localFileChooser);	//I don't know if I've ever seen anyone do this before.  It's a little weird, but it's quick and accessible.
		else
		{
			openFileDialogItem = new JMenuItem("Open...");
			openFileDialogItem.addActionListener(this);
			fileMenu.add(openFileDialogItem);
		}
		menubar.add(fileMenu);
		JPanel localCodePanel = new JPanel(new BorderLayout());
		localCodePanel.add(menubar, BorderLayout.NORTH);
		localCodePanel.add(new JScrollPane(numberedTextPanel, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS));
		
		tabs.add(localCodePanel, "Local Code");
		
		mainPanel = new JPanel(new BorderLayout());
		mainPanel.add(tabs, BorderLayout.CENTER);
		
		generateButton = new JButton("Generate code");
		generateButton.addActionListener(this);
		
		JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		buttonPanel.add(generateButton);
		
		challengeToolsButton = new JButton("Challenge tools");
		challengeToolsButton.addActionListener(this);
		challengeToolsButton.setMnemonic(KeyEvent.VK_C);
		challengeToolsButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(challengeToolsButton);
		
		testButton = new JButton("Run Tests");
		testButton.addActionListener(this);
		testButton.setMnemonic(KeyEvent.VK_R);
		testButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(testButton);
		
		saveButton = new JButton("Save");
		saveButton.addActionListener(this);
		saveButton.setMnemonic(KeyEvent.VK_S);
		saveButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(saveButton);
		
		killButton = new JButton("Kill Process");
		killButton.addActionListener(this);
		killButton.setMnemonic(KeyEvent.VK_K);
		killButton.setDisplayedMnemonicIndex(0);
		buttonPanel.add(killButton);
		
		lineLabel = new JLabel("Line: 0");
		lineLabel.setForeground(Color.white);
		lineLabel.setFont(new Font("Monospaced", Font.BOLD, 14));
		buttonPanel.add(lineLabel);
		
		mainPanel.add(buttonPanel, BorderLayout.SOUTH);
		
		File f = new File(tempdir, "snippets.dat");
		if (f.exists())
		{
			try
			{
				ObjectInputStream in = new ObjectInputStream(new BufferedInputStream(new FileInputStream(f)));
				baseCategory = (Category)in.readObject();
				in.close();
			}
			catch (IOException ex)
			{
			}
			catch (ClassNotFoundException ex)
			{
			}
		}
		if (baseCategory == null)
			baseCategory = new Category("");
		baseCategory.propagateActionListener(this);
		
		nameField = new JTextField();
		nameField.addActionListener(this);
		categoryField = new JTextField(12);
		categoryField.addActionListener(this);
		snippetButton = new JButton("Record Snippet");
		snippetButton.addActionListener(this);
		snippetPanel = new JPanel(new GridLayout(3, 2));
		JLabel temp = new JLabel("Snippet name:");
		temp.setForeground(Color.white);
		snippetPanel.add(temp);
		snippetPanel.add(nameField);
		temp = new JLabel("Category:");
		temp.setForeground(Color.white);
		snippetPanel.add(temp);
		snippetPanel.add(categoryField);
		snippetPanel.add(new JLabel());
		snippetPanel.add(snippetButton);
		
		replaceField = new JTextField();
		replaceField.addActionListener(this);
		replaceWithField = new JTextField();
		replaceWithField.addActionListener(this);
		replaceButton = new JButton("Replace");
		replaceButton.addActionListener(this);
		replacePanel = new JPanel(new GridLayout(3, 2));
		temp = new JLabel("Replace:");
		temp.setForeground(Color.white);
		replacePanel.add(temp);
		replacePanel.add(replaceField);
		temp = new JLabel("With:");
		temp.setForeground(Color.white);
		replacePanel.add(temp);
		replacePanel.add(replaceWithField);
		replacePanel.add(new JLabel());
		replacePanel.add(replaceButton);
		
		varReplaceField = new JTextField();
		varReplaceField.addActionListener(this);
		varReplaceWithField = new JTextField();
		varReplaceWithField.addActionListener(this);
		varReplaceButton = new JButton("Change");
		varReplaceButton.addActionListener(this);
		varReplacePanel = new JPanel(new GridLayout(3, 2));
		temp = new JLabel("Change:");
		temp.setForeground(Color.white);
		varReplacePanel.add(temp);
		varReplacePanel.add(varReplaceField);
		temp = new JLabel("To:");
		temp.setForeground(Color.white);
		varReplacePanel.add(temp);
		varReplacePanel.add(varReplaceWithField);
		varReplacePanel.add(new JLabel());
		varReplacePanel.add(varReplaceButton);
		undo = new UndoManager();
	}
	
	/**
	 *	Saves the snippet database to disk in a file called "snippets.dat".
	 **/
	protected void saveSnippets()
	{
		try
		{
			ObjectOutputStream out = new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(new File(tempdir, "snippets.dat"))));
			out.writeObject(baseCategory);
			out.close();
		}
		catch (IOException ex)
		{
			System.err.println("Couldn't write snippets");
		}
	}
	
	/**
	 *	Handles events of all the buttons, menus, and so forth.
	 **/
	public void actionPerformed(ActionEvent e)
	{
		if (e.getSource() == generateButton)
		{
			if (component != null && language != null)
				textArea.setText(language.getSkeleton(component));
		}
		else if (e.getSource() == testButton)
		{
			testingPane.setText("");
			tabs.setSelectedIndex(tabs.indexOfTab("Test Results"));
			try
			{
				saveLocal();
				if (compileLocal())
				{
					testingPane.setText(testingPane.getText() + "Compiled\n\n");
					
					if (proc == null || proc.isDone())
					{
						Process p = Runtime.getRuntime().exec(language.getRunCommand(component, tempdir), null, tempdir);
						proc = new ProcessContainer(p, testingPane);
						proc.start();
					}
					else
					{
						testingPane.setText(testingPane.getText() + "Error: Can't start new process while another is running.\n");
					}
				}
				else
					testingPane.setText(testingPane.getText() + "Compiling errors\n");
			}
			catch (Throwable t)
			{
				testingPane.setText(testingPane.getText() + "My error: " + t + "\n");
				t.printStackTrace();
			}
		}
		else if (e.getSource() == saveButton)
		{
			try
			{
				saveLocal();
			}
			catch (Throwable t)
			{
				lineLabel.setText("Couldn't save!");
			}
		}
		else if (e.getSource() == killButton)
		{
			if (proc != null &! proc.isDone())
				proc.kill();
		}
		else if (e.getSource() == localFileChooser)
		{
			fileMenu.setSelected(false);
			fileMenu.setPopupMenuVisible(false);
			File f = localFileChooser.getSelectedFile();
			try
			{
				BufferedReader inFile = new BufferedReader(new FileReader(f));
				String text = "";
				String line;
				while ((line = inFile.readLine()) != null)
					text += line + "\n";
				inFile.close();
				if (f.getName().toLowerCase().endsWith(".cs"))
					localCodePane.setContentType("text/" + CSharpLanguage.CSHARP_LANGUAGE.getName());
				else if (f.getName().toLowerCase().endsWith(".java"))
					localCodePane.setContentType("text/" + JavaLanguage.JAVA_LANGUAGE.getName());
				else if (f.getName().toLowerCase().endsWith(".vb"))
					localCodePane.setContentType("text/" + VBLanguage.VB_LANGUAGE.getName());
				else
					localCodePane.setContentType("text/" + CPPLanguage.CPP_LANGUAGE.getName());
				localCodePane.setText(text);
			}
			catch (IOException ex)
			{
				localCodePane.setText("IOException thrown!");
			}
		}
		else if (e.getSource() == openFileDialogItem)
		{
			localFileChooser.showOpenDialog(null);
		}
		else if (e.getSource() == snippetButton || e.getSource() == nameField || e.getSource() == categoryField)
		{
			Snippet s = new Snippet(codeSelection, nameField.getText());
			baseCategory.add(s, categoryField.getText());
			s.addActionListener(this);
			if (popup != null)
				popup.setVisible(false);
			saveSnippets();
		}
		else if (e.getSource() == replaceButton || e.getSource() == replaceField || e.getSource() == replaceWithField)
		{
			if (replaceField.getText().length() > 0 && replaceWithField.getText().length() > 0)
			{
				if (textArea.getSelectedText() == null)
					textArea.setText(textArea.getText().replaceAll(replaceField.getText(), replaceWithField.getText()));
				else
					textArea.replaceSelection(textArea.getSelectedText().replaceAll(replaceField.getText(), replaceWithField.getText()));
				if (popup != null)
					popup.setVisible(false);
			}
		}
		else if (e.getSource() == varReplaceButton || e.getSource() == varReplaceField || e.getSource() == varReplaceWithField)
		{
			if (varReplaceField.getText().length() > 0 && varReplaceWithField.getText().length() > 0)
			{
				if (textArea.getSelectedText() == null)
					textArea.setText(textArea.getText().replaceAll("([^a-zA-Z0-9_])" + varReplaceField.getText() + "([^a-zA-Z0-9_])", "$1" + varReplaceWithField.getText() + "$2"));
				else
					textArea.replaceSelection(textArea.getSelectedText().replaceAll("([^a-zA-Z0-9_])" + varReplaceField.getText() + "([^a-zA-Z0-9_])", "$1" + varReplaceWithField.getText() + "$2"));
				if (popup != null)
					popup.setVisible(false);
			}
		}
		else if (e.getSource() == challengeToolsButton)
		{
			HistoryLookup.getFrame().show();
		}
		else if (e.getSource() instanceof Snippet)
		{
			if ((e.getModifiers()&ActionEvent.META_MASK) != 0)
				((Snippet)e.getSource()).delete();
			else if (currentPopupThing != null)
				currentPopupThing.replaceSelection(e.getSource().toString());
		}
	}
	
	/**
	 *	Saves the current file to disk in the configured local directory with the appropriate name.
	 **/
	protected void saveLocal() throws IOException
	{
		if (!tempdir.exists())
			tempdir.mkdir();
		PrintWriter outFile = new PrintWriter(new FileWriter(new File(tempdir, language.getFileName(component))));
		outFile.println(textArea.getText());
		if (language != null && language.getId() == CPPLanguage.ID)
		{
			outFile.println("int main() {");
			outFile.println("int time;");
			outFile.println("\tbool errors = false;");
			outFile.println("\t");
			for (int i=0; i<component.getTestCases().length; i++)
			{
				outFile.println("\ttime = test" + i + "();");
				outFile.println("\tif (time < 0)");
				outFile.println("\t\terrors = true;");
				outFile.println("\t");
			}
			outFile.println("\tif (!errors)");
			outFile.println("\t\tcout <<\"You\'re a stud (at least on the example cases)!\" <<endl;");
			outFile.println("\telse");
			outFile.println("\t\tcout <<\"Some of the test cases had errors.\" <<endl;");
			outFile.println("}");
		}
		outFile.close();
	}
	
	/**
	 *	Compiles the saved file locally.
	 **/
	protected boolean compileLocal() throws Exception
	{
		if (proc == null || proc.isDone())
		{
			Process p = Runtime.getRuntime().exec(language.getCompileCommand(component), null, tempdir);
			proc = new ProcessContainer(p, testingPane);
			proc.start();
			p.waitFor();
			return proc.endVal() == 0;
		}
		else
		{
			testingPane.setText(testingPane.getText() + "Error: Can't compile while another process is running\n");
			return false;
		}
	}
	
	/**
	 *	Returns the magic KawigiEdit panel. 
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public JPanel getEditorPanel()
	{
		return mainPanel;
	}
	
	/**
	 *	Returns the text in the editor.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public String getSource()
	{
		return textArea.getText();
	}
	
	/**
	 *	Sets the text in the editor.
	 *	
	 *	This implementation will ignore the request if the source provided is empty.
	 *	This is to maintain auto-generated code.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void setSource(String source)
	{
		if (source.length() > 0)
			textArea.setText(source);
	}
	
	/**
	 *	Empties the text pane.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void clear()
	{
		textArea.setText("");
	}
	
	/**
	 *	Enables/disables the text pane.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void setTextEnabled(Boolean b)
	{
		textArea.setEnabled(b.booleanValue());
	}
	
	/**
	 *	Notifies the editor of a new problem being opened, or the language being changed, or whatever.
	 *	
	 *	This resets some of the state, including the undo manager.  If the editor has no text yet, this
	 *	method will also generate skeleton and test code for the problem.  Eventually in the right language, even.
	 **/
	public void setProblemComponent(ProblemComponentModel component, Language lang, com.topcoder.shared.problem.Renderer renderer)
	{
		language = ExtendedLanguageFactory.getLanguage(lang);
		textArea.setContentType("text/" + language.getName());
		textArea.getStyledDocument().addUndoableEditListener(this);
		undo.discardAllEdits();
		if (textArea.getText().length() == 0)
			textArea.setText(language.getSkeleton(this.component = component.getComponent()));
	}
	
	/**
	 *	Clears the text pane for a new problem or something.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void startUsing()
	{
		textArea.setText("");
	}
	
	/**
	 *	Doesn't do anything.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void stopUsing()
	{
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	Specifically listens for ctrl+z and ctrl+y for undo and redo, respectively.
	 **/
	public void keyPressed(KeyEvent e)
	{
		if (e.getKeyCode() == KeyEvent.VK_Z && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
		{
			try
			{
				undo.undo();
			}
			catch (CannotUndoException ex)
			{
			}

		}
		if (e.getKeyCode() == KeyEvent.VK_Y && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
		{
			try
			{
				undo.redo();
			}
			catch (CannotRedoException ex)
			{
			}
		}
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	Specifically listens for ctrl+i to popup the snippet popup menu.
	 **/
	public void keyReleased(KeyEvent e)
	{
		if (e.getKeyCode() == KeyEvent.VK_I && (e.getModifiersEx()&InputEvent.CTRL_DOWN_MASK) != 0)
		{
			JPopupMenu popup = getPopup(textArea);
			Point pt = textArea.getCaret().getMagicCaretPosition();
			if (pt == null)
				pt = new Point();
			popup.show(textArea, pt.x, pt.y);
		}
	}
	
	/**
	 *	Listens for Key events on the text pane.
	 *	
	 *	This handles the auto-indent.
	 **/
	public void keyTyped(KeyEvent e)
	{
		if (e.getKeyChar() == '\n')
		{
			String text = textArea.getText().replaceAll("\\r", "");
			String line = "";
			String indentation = "";
			int lines = 0;
			for (int ind = -1; ind < textArea.getCaretPosition(); ind = text.indexOf('\n', ind+1))
			{
				if (ind < 0 && lines != 0)
					break;
				lines++;
				indentation = (line.trim().length() == 0) ? line : line.substring(0, line.indexOf(line.trim()));
				line = (text.indexOf('\n', ind+1) < 0) ? text.substring(ind+1) : text.substring(ind+1, text.indexOf('\n', ind+1));
			}
			if ((e.getModifiersEx()&InputEvent.SHIFT_DOWN_MASK) != 0)
				indentation = "\n" + indentation;
			textArea.replaceSelection(indentation);
		}
		textArea.repaint();
	}
	
	/**
	 *	Listens for changes in caret position on the text pane.
	 *	
	 *	Updates the line number displayed on the bottom-right.
	 **/
	public void caretUpdate(CaretEvent e)
	{
		int lines = 0;
		String text = textArea.getText().replaceAll("\\r", "");
		for (int ind = -1; ind<e.getDot(); ind = text.indexOf('\n', ind+1))
		{
			if (ind < 0 && lines != 0)
				break;
			lines++;
		}
		lineLabel.setText("Line: " + lines);
	}
	
	/**
	 *	Brings up a configure dialog to set options in the editor plugin.
	 *	
	 *	If I were to pick just one thing to complain about in the plugin specification,
	 *	it would be the description on how to do this.  I found it completely vague,
	 *	and after reading still wasn't sure what it was supposed to do.  Originally,
	 *	this method actually froze or something, or otherwise didn't work, until I
	 *	made the configure dialog modal.  On the side, we are encouraged to use TopCoder's
	 *	preferences class, but it is completely undocumented and the package it is
	 *	found in is not specified.  So I hacked up an old program I wrote that reads
	 *	a jar and uses the jar classes as well as reflection classes to chart the
	 *	classes in the jar on a JTree, and looked for about half an hour for it.
	 *	Anyways, someone should write a tutorial on using configure() or something.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void configure()
	{
		//with no real available documentation, the best way to figure out this part is:
		//	1. Reflection
		//	2. Trial/Error
		//	3. Go to the Round Tables and whine to Pops until someone explains it
		//As such, I will probably include my Reflector to this package just for other
		//plugin writers to use.
		ConfigPanel frame = new ConfigPanel();
		frame.pack();
		frame.show();
		CPPView.initColors();
		CSharpView.initColors();
		JavaView.initColors();
		VBView.initColors();
		LocalPreferences prefs = LocalPreferences.getInstance();
		textArea.setForeground(prefs.getColor("kawigi.editor.foreground"));
		textArea.setCaretColor(textArea.getForeground());
		textArea.setBackground(prefs.getColor("kawigi.editor.background"));
		textArea.setFont(new Font(prefs.getProperty("kawigi.editor.font"), Font.PLAIN, Integer.parseInt(prefs.getProperty("kawigi.editor.font.size"))));
		localCodePane.setBackground(textArea.getBackground());
		localCodePane.setForeground(textArea.getForeground());
		localCodePane.setCaretColor(textArea.getCaretColor());
		localCodePane.setFont(textArea.getFont());
		testingPane.setForeground(prefs.getColor("kawigi.testing.foreground"));
		testingPane.setCaretColor(testingPane.getForeground());
		testingPane.setBackground(prefs.getColor("kawigi.testing.background"));
		testingPane.setFont(new Font(prefs.getProperty("kawigi.testing.font"), Font.PLAIN, Integer.parseInt(prefs.getProperty("kawigi.testing.font.size"))));
		useStdUI = prefs.isTrue("kawigi.stdui");
		tempdir = new File(prefs.getProperty("kawigi.localpath"));
		if (!tempdir.exists())
			tempdir.mkdir();
	}
	
	/**
	 *	Verifies or sets several properties used by parts of the editor to set up.
	 *	
	 *	For awhile, I was using this, but the constructor pretty much checks to make 
	 *	I have all my configurations intact.
	 *	
	 *	From the TopCoder plugin interface.
	 **/
	public void install()
	{
	}
	
	/**
	 *	Stores away edits so they can be undone later.
	 **/
	public void undoableEditHappened(UndoableEditEvent e)
	{
		//Remember the edit and update the menus
		undo.addEdit(e.getEdit());
	}
	
	/**
	 *	Creates, saves, and returns a popup menu for the given JTextPane to display.
	 *	
	 *	This has the snippet menu, search/replace stuff, and (if text is selected)
	 *	the add-snippet thingy.  I know the UI is non-standard, maybe I'll provide
	 *	a way to configure it to use dialogs instead of funny panels on a popup menu.
	 **/
	private JPopupMenu getPopup(NoWrapJTextPane pane)
	{
		currentPopupThing = pane;
		popup = new JPopupMenu();
		popup.add(baseCategory);
		if (pane.getSelectedText() != null)
		{
			JMenu weirdMenu = new JMenu("Add Snippet");
			weirdMenu.add(snippetPanel);
			popup.add(weirdMenu);
			String code = pane.getSelectedText();
			codeSelection = code;
			if (code.indexOf('(') >= 0)
			{
				String name = code.substring(0, code.indexOf('(')).trim();
				if (name.indexOf('\n') >= 0)
					name = name.substring(name.lastIndexOf('\n')+1).trim();
				if (name.indexOf(' ') >= 0)
					name = name.substring(name.lastIndexOf(' ')+1).trim();
				nameField.setText(name);
			}
			else
				nameField.setText("");
		}
		JMenu replaceMenu = new JMenu("Search/Replace");
		replaceMenu.add(replacePanel);
		popup.add(replaceMenu);
		JMenu varReplaceMenu = new JMenu("Change var name");
		varReplaceMenu.add(varReplacePanel);
		popup.add(varReplaceMenu);
		return popup;
	}
	
	/**
	 *	Listens to mouse events on the text pane, so it knows when to create the
	 *	popup menu.
	 **/
	public void mousePressed(MouseEvent e)
	{
		if (e.isPopupTrigger())
		{
			JPopupMenu popup = getPopup((NoWrapJTextPane)e.getComponent());
			popup.show(e.getComponent(), e.getX(), e.getY());
		}
	}
	
	/**
	 *	Listens to mouse events on the text pane, so it knows when to create the
	 *	popup menu.
	 **/
	public void mouseReleased(MouseEvent e)
	{
		if (e.isPopupTrigger())
		{
			JPopupMenu popup = getPopup((NoWrapJTextPane)e.getComponent());
			popup.show(e.getComponent(), e.getX(), e.getY());
		}
	}
	
	/**
	 *	Required by the MouseListener interface.
	 **/
	public void mouseClicked(MouseEvent e){}
	
	/**
	 *	Required by the MouseListener interface.
	 **/
	public void mouseEntered(MouseEvent e){}
	
	/**
	 *	Required by the MouseListener interface.
	 **/
	public void mouseExited(MouseEvent e){}
	
	/**
	 *	Run the plugin as a standalone application.
	 *	
	 *	This is mostly for view debugging, since it won't do anything exciting, and
	 *	many of the features won't completely work (because the plugin methods won't
	 *	be called).
	 **/
	public static void main(String[] args)
	{
		JFrame frame = new JFrame("KawigiEdit 1.0");
		frame.getContentPane().add(new KawigiEdit().getEditorPanel());
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setSize(500, 400);
		frame.show();
	}
}
